// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/gui/gui_ascii_renderer.hh"

#include "nv/core/logging.hh"

using namespace nv;
using namespace nv::gui;


struct ascii_render_data : public render_data
{
	ascii_render_data() {}
	bool   clear;
	bool   border;
	uchar8 border_chars[8];
	uint32 border_color;
	uint32 text_color;
};


ascii_renderer::ascii_renderer( terminal* t )
	: m_terminal(t)
{

}

void ascii_renderer::redraw( element* e, uint32 )
{
	ascii_render_data* er = nullptr;
	if ( e->m_render_data == nullptr )
	{
		er = new ascii_render_data;
		er->border     = false;
		er->clear      = false;
		er->text_color = 0;
		e->m_render_data = er;
	}
	else 
		er = static_cast< ascii_render_data* >( e->m_render_data );

	rectangle abs = e->m_absolute;
	if ( abs != get_area() )
	{
		er->clear = true;
		int color = 0;
		string128 path;
		const char* stext[] = { nullptr, "selected", "hover" };
		const char* selector = stext[0];
		if ( e->m_flags[HOVER] )    selector = stext[2];
		if ( e->m_flags[SELECTED] ) selector = stext[1];

		
		m_style.get( e, "ascii_color", selector, color );
		er->text_color = uint32( color );
		er->border     = false;
		if ( m_style.get( e, "ascii_border", selector, path ) )
		{
			er->border = true;
			er->border_color = er->text_color;
			int border_color = 0;
			if ( m_style.get( e, "ascii_border_color", selector, border_color ) )
				er->border_color = uint32( border_color );
			for ( uint32 i = 0; i < 8 && i < path.length(); i++ )
				er->border_chars[i] = static_cast< uchar8 >( path[i] );
		}
	}
}

void ascii_renderer::draw( element* e )
{
	ascii_render_data* er = static_cast< ascii_render_data* >( e->m_render_data );
	rectangle abs = e->m_absolute;
	if ( er->clear ) m_terminal->clear( abs );
	if ( er->border )
	{
		for ( int x = 0; x < abs.get_width(); ++x )
		{
			m_terminal->print( position( abs.ul.x + x, abs.ul.y ), er->border_color, er->border_chars[0] );
			m_terminal->print( position( abs.ul.x + x, abs.lr.y ), er->border_color, er->border_chars[1] );
		}

		for ( int y = 0; y < abs.get_height(); ++y )
		{
			m_terminal->print( position( abs.ul.x, abs.ul.y + y ), er->border_color, er->border_chars[2] );
			m_terminal->print( position( abs.lr.x, abs.ul.y + y ), er->border_color, er->border_chars[3] );
		}

		m_terminal->print( abs.ul,   er->border_color, er->border_chars[4] );
		m_terminal->print( abs.ur(), er->border_color, er->border_chars[5] );
		m_terminal->print( abs.ll(), er->border_color, er->border_chars[6] );
		m_terminal->print( abs.lr,   er->border_color, er->border_chars[7] );
		m_terminal->update();
	}
	if ( !e->m_text.empty() )
	{
		position p = abs.ul;
		for ( char c : e->m_text )
		{
			m_terminal->print( p, er->text_color, static_cast< unsigned char >( c ) );
			++p.x;
		}
	}
}

void ascii_renderer::draw()
{
	m_terminal->update();
}

void nv::gui::ascii_renderer::on_style_change( element* e )
{
	m_style.load_flags( e );
}

void nv::gui::ascii_renderer::on_hover_change( element* e )
{
	if ( e->m_flags[DIRTY_HOVER] ) e->m_flags[DIRTY] = true;
}

void nv::gui::ascii_renderer::on_select_change( element* e )
{
	if ( e->m_flags[DIRTY_SELECT] ) e->m_flags[DIRTY] = true;
}

rectangle ascii_renderer::get_area() const
{
	return rectangle().dim( m_terminal->get_size() );
}

ascii_renderer::~ascii_renderer()
{

}
