// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file type_erasure.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief type erasure
 */


#ifndef NV_STL_TYPE_ERASURE_HH
#define NV_STL_TYPE_ERASURE_HH

#include <nv/common.hh>
#include <nv/stl/iterator.hh>

namespace nv
{

	template < typename T >
	class raw_data_iterator : public iterator< random_access_iterator_tag, T, ptrdiff_t, T*, T& >
	{
	public:
		typedef random_access_iterator_tag iterator_category;
		typedef T                          value_type;
		typedef ptrdiff_t                  difference_type;
		typedef T*                         pointer;
		typedef T&                         reference;

		raw_data_iterator( void* data, uint32 stride = 0, uint32 offset = 0 )
		{
			m_current = static_cast<uint8*>( data ) + offset;
			m_stride = stride;
		}

		inline reference operator*() const { return *reinterpret_cast<pointer>( m_current ); }
		inline pointer operator->() const { return reinterpret_cast<pointer>( m_current ); }

		inline raw_data_iterator& operator++() { m_current += stride; return *this; }
		inline raw_data_iterator operator++( int ) { raw_data_iterator ri( *this ); m_current += stride; return ri; }
		inline raw_data_iterator& operator--() { m_current -= stride; return *this; }
		inline raw_data_iterator operator--( int ) { raw_data_iterator ri( *this ); m_current -= stride; return ri; }

		inline raw_data_iterator operator+( difference_type n ) const { return raw_data_iterator( m_current + n * m_stride, m_stride ); }
		inline raw_data_iterator& operator+=( difference_type n ) { m_current += n * m_stride; return *this; }
		inline raw_data_iterator operator-( difference_type n ) const { return raw_data_iterator( m_current - n * m_stride, m_stride ); }
		inline raw_data_iterator& operator-=( difference_type n ) { m_current -= n * m_stride; return *this; }
		constexpr reference operator[]( difference_type n ) const { return *reinterpret_cast<pointer>( m_current + n * m_stride ); }

		const uint8* raw_data() const { return m_current; }
		uint32 stride() const { return m_stride; }
	protected:
		uint8* m_current;
		uint32 m_stride;
	};


	template < typename T >
	class const_raw_data_iterator : public iterator< random_access_iterator_tag, T, ptrdiff_t, const T*, const T& >
	{
	public:
		typedef random_access_iterator_tag iterator_category;
		typedef T                          value_type;
		typedef ptrdiff_t                  difference_type;
		typedef const T*                   pointer;
		typedef const T&                   reference;

		constexpr const_raw_data_iterator( const raw_data_iterator< T >& it )
		{
			m_current = it.raw_data();
			m_stride  = it.stride();
		}

		constexpr const_raw_data_iterator( void* data, uint32 stride = 0, uint32 offset = 0 )
		{
			m_current = static_cast<const uint8*>( data ) + offset;
			m_stride  = stride;
		}
		constexpr reference operator*() const { return *reinterpret_cast<pointer>( m_current ); }
		constexpr pointer operator->() const { return reinterpret_cast<pointer>( m_current ); }

		constexpr const_raw_data_iterator& operator++() { m_current += stride; return *this; }
		constexpr const_raw_data_iterator operator++( int ) { const_raw_data_iterator ri( *this ); m_current += stride; return ri; }
		constexpr const_raw_data_iterator& operator--() { m_current -= stride; return *this; }
		constexpr const_raw_data_iterator operator--( int ) { const_raw_data_iterator ri( *this ); m_current -= stride; return ri; }

		constexpr const_raw_data_iterator operator+( difference_type n ) const { return const_raw_data_iterator( m_current + n * m_stride, m_stride ); }
		constexpr const_raw_data_iterator& operator+=( difference_type n ) { m_current += n * m_stride; return *this; }
		constexpr const_raw_data_iterator operator-( difference_type n ) const { return const_raw_data_iterator( m_current - n * m_stride, m_stride ); }
		constexpr const_raw_data_iterator& operator-=( difference_type n ) { m_current -= n * m_stride; return *this; }
		constexpr reference operator[]( difference_type n ) const { return *reinterpret_cast<pointer>( m_current + n * m_stride ); }

		const uint8* raw_data() const { return m_current; }
		uint32 stride() const { return m_stride; }
	protected:
		const uint8* m_current;
		uint32 m_stride;
	};

	template < typename T >
	constexpr bool operator==( const raw_data_iterator< T >& lhs, const raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() == rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator!=( const raw_data_iterator< T >& lhs, const raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() != rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator<( const raw_data_iterator< T >& lhs, const raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() < rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator>( const raw_data_iterator< T >& lhs, const raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() > rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator<=( const raw_data_iterator< T >& lhs, const raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() <= rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator>=( const raw_data_iterator< T >& lhs, const raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() >= rhs.raw_data();
	}
	template < typename T >
	constexpr typename raw_data_iterator< T >::difference_type operator-( const raw_data_iterator< T >& lhs, const raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return ( lhs.raw_data() - rhs.raw_data() ) / lhs->m_stride;
	}
	template < typename T >
	constexpr raw_data_iterator< T > operator+( typename raw_data_iterator< T >::difference_type n, const raw_data_iterator< T >& iter )
	{
		return raw_data_iterator< T >( iter.raw_data() + iter.stride() * n, iter.stride() );
	}

	template < typename T >
	constexpr bool operator==( const const_raw_data_iterator< T >& lhs, const const_raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() == rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator!=( const const_raw_data_iterator< T >& lhs, const const_raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() != rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator<( const const_raw_data_iterator< T >& lhs, const const_raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() < rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator>( const const_raw_data_iterator< T >& lhs, const const_raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() > rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator<=( const const_raw_data_iterator< T >& lhs, const const_raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() <= rhs.raw_data();
	}
	template < typename T >
	constexpr bool operator>=( const const_raw_data_iterator< T >& lhs, const const_raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return lhs.raw_data() >= rhs.raw_data();
	}
	template < typename T >
	constexpr typename const_raw_data_iterator< T >::difference_type operator-( const const_raw_data_iterator< T >& lhs, const const_raw_data_iterator< T >& rhs )
	{
		NV_ASSERT( lhs.stride() == rhs.stride(), "stride failure!" );
		return ( lhs.raw_data() - rhs.raw_data() ) / lhs->m_stride;
	}
	template < typename T >
	constexpr const_raw_data_iterator< T > operator+( typename const_raw_data_iterator< T >::difference_type n, const const_raw_data_iterator< T >& iter )
	{
		return const_raw_data_iterator< T >( iter.raw_data() + iter.stride() * n, iter.stride() );
	}
}

#endif // NV_STL_TYPE_ERASURE_HH



