// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_STL_STRING_LITERAL_STRING_HH
#define NV_STL_STRING_LITERAL_STRING_HH

#include <nv/stl/string/common.hh>
#include <nv/stl/string/string_base.hh>

namespace nv
{

	class literal_string : public string_base< array_view< char > >
	{
		constexpr literal_string( const char* str, size_t len ) : this_type( str, len ) {}
	public:
		typedef string_base< array_view< char > > this_type;
		friend constexpr literal_string operator "" _ls( const char* str, size_t len );

		template< size_t N >
		constexpr literal_string( char( &s )[N] ) : this_type( s, N - 1 ) {}
		template< size_t N >
		constexpr literal_string( const char( &s )[N] ) : this_type( s, N - 1 ) {}
	};

	constexpr literal_string operator "" _ls( const char* str, size_t len )
	{
		return literal_string( str, len );
	}

	template < typename H >
	class hashed_literal_string;
	using hashed_literal_string_32 = hashed_literal_string< uint32 >;
	using hashed_literal_string_64 = hashed_literal_string< uint64 >;

	template < typename H >
	class hashed_literal_string : public string_base< array_view< char > >
	{
		constexpr hashed_literal_string( const char* str, size_t len, H hash_code ) 
			: this_type( str, len ), m_hash( hash_code ) {}
	public:
		typedef string_base< array_view< char > > this_type;

		template< size_t N >
		constexpr hashed_literal_string( char( &s )[N] )
			: this_type( s, N - 1 ), m_hash( detail::fnv_hash<H>::hash( s, N - 1 ) ) {}
		template< size_t N >
		constexpr hashed_literal_string( const char( &s )[N] )
			: this_type( s, N - 1 ), m_hash( detail::fnv_hash<H>::hash( s, N - 1 ) ) {}

		template < typename H2 >
		constexpr H get_hash() const
		{
			static_assert( is_same< H, H2 >::value, "32/64 bit hash mismatch on hash_literal_string!" );
			return m_hash;
		}

		friend constexpr hashed_literal_string_32 operator "" _hls32( const char* str, size_t len );
		friend constexpr hashed_literal_string_64 operator "" _hls64( const char* str, size_t len );
	protected:
		H m_hash;
	};

	constexpr hashed_literal_string_32 operator "" _hls32( const char* str, size_t len )
	{
		return hashed_literal_string_32( str, len, detail::fnv_hash< uint32 >::hash( str, len ) );
	}

	constexpr hashed_literal_string_64 operator "" _hls64( const char* str, size_t len )
	{
		return hashed_literal_string_64( str, len, detail::fnv_hash< uint64 >::hash( str, len ) );
	}

	template< typename H >
	struct hash< hashed_literal_string_32, H >
	{
		static_assert( is_same< uint32, H >::value, "hashed_literal_string_32 used with non-32 bit hash!" );
		static constexpr H get( const hashed_literal_string_32& value )
		{
			return value.get_hash< H >();
		}
		constexpr H operator()( const hashed_literal_string_32& value ) const { return get( value ); }
	};

	template< typename H >
	struct hash< hashed_literal_string_64, H >
	{
		static_assert( is_same< uint64, H >::value, "hashed_literal_string_64 used with non-64 bit hash!" );
		static constexpr H get( const hashed_literal_string_64& value )
		{
			return value.get_hash< H >();
		}
		constexpr H operator()( const hashed_literal_string_64& value ) const { return get( value ); }
	};

}

using nv::operator "" _ls;
using nv::operator "" _hls32;
using nv::operator "" _hls64;

#endif // NV_STL_STRING_LITERAL_STRING_HH
