// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file relational.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief relational math ops
*/

#ifndef NV_STL_MATH_RELATIONAL_HH
#define NV_STL_MATH_RELATIONAL_HH

#include <nv/stl/math/common.hh>

namespace nv
{

	namespace math
	{

		namespace detail
		{
			template < typename T >
			struct make_bool_vec;

			template < typename T >
			struct make_bool_vec< tvec2< T > >
			{
				typedef tvec2< bool > type;
			};

			template < typename T >
			struct make_bool_vec< tvec3< T > >
			{
				typedef tvec3< bool > type;
			};

			template < typename T >
			struct make_bool_vec< tvec4< T > >
			{
				typedef tvec4< bool > type;
			};

		}

		template < typename T >
		using bool_vec_t = typename detail::make_bool_vec<T>::type;

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline bool_vec_t<T> less_than( const T& lhs, const T& rhs )
		{
			bool_vec_t<T> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( lhs ); ++i )
				result[i] = lhs[i] < rhs[i];

			return result;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline bool_vec_t<T> less_than_equal( const T& lhs, const T& rhs )
		{
			bool_vec_t<T> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( lhs ); ++i )
				result[i] = lhs[i] <= rhs[i];
			return result;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline bool_vec_t<T> greater_than( const T& lhs, const T& rhs )
		{
			bool_vec_t<T> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( lhs ); ++i )
				result[i] = lhs[i] > rhs[i];
			return result;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline bool_vec_t<T> greater_than_equal( const T& lhs, const T& rhs )
		{
			bool_vec_t<T> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( lhs ); ++i )
				result[i] = lhs[i] >= rhs[i];
			return result;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline bool_vec_t<T> equal( const T& lhs, const T& rhs )
		{
			bool_vec_t<T> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( lhs ); ++i )
				result[i] = lhs[i] == rhs[i];
			return result;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline bool_vec_t<T> not_equal( const T& lhs, const T& rhs )
		{
			bool_vec_t<T> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( lhs ); ++i )
				result[i] = lhs[i] != rhs[i];
			return result;
		}

		template < typename T, typename enable_if< is_bool_vec<T>::value >::type* = nullptr >
		inline bool any( const T& v )
		{
			bool result = false;
			for ( component_count_t i = 0; i < component_count( v ); ++i )
				result = result || v[i];
			return result;
		}

		template < typename T, typename enable_if< is_bool_vec<T>::value >::type* = nullptr >
		inline bool all( const T& v )
		{
			bool result = true;
			for ( component_count_t i = 0; i < component_count( v ); ++i )
				result = result && v[i];
			return result;
		}

		template < typename T, typename enable_if< is_bool_vec<T>::value >::type* = nullptr >
		inline T not_( const T& v )
		{
			T result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( v ); ++i )
				result[i] = !v[i];
			return result;
		}

	}

}

#endif // NV_STL_MATH_RELATIONAL_HH
