// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file basic.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief basic math ops
*/

#ifndef NV_STL_MATH_BASIC_HH
#define NV_STL_MATH_BASIC_HH

#include <nv/stl/math/common.hh>
#include <nv/stl/math/relational.hh>

namespace nv
{

	namespace math
	{
		namespace detail
		{

			template <
				typename T,
				bool Signed = numeric_limits<T>::is_signed
			>
			struct abs_helper;

			template < typename T >
			struct abs_helper< T, false >
			{
				constexpr static T call( T x )
				{
					return x;
				}
			};

			template < typename T >
			struct abs_helper< T, true >
			{
				constexpr static T call( T x )
				{
					return x >= T( 0 ) ? x : -x;
				}
			};

			template <
				typename T,
				typename Mix = T,
				bool MixBool = is_same< value_type_t<Mix>, bool >,
				bool TIsVector = is_vec<T>::value,
				bool MixIsVector = is_vec<Mix>::value >
			struct mix_helper
			{
				inline static T call( const T& a, const T& b, const Mix& m )
				{
					return a + m * ( b - a );
				}
			};

			template < typename T, bool IsVector >
			struct mix_helper < T, bool, true, IsVector, false >
			{
				inline static T call( const T& a, const T& b, bool m )
				{
					return m ? b : a;
				}
			};

			template < typename T, typename U >
			struct mix_helper < T, U, true, true, true >
			{
				inline static T call( const T& a, const T& b, const U& m )
				{
					T result( ctor::uninitialize );
					for ( component_count_t i = 0; i < component_count( a ); ++i )
						result[i] = m[i] ? b[i] : a[i];
					return result;
				}
			};

			template < typename T, typename U = value_type_t<T>, bool IsFloating = is_floating_point< U >::value, bool IsSigned = is_signed< U >::value >
			struct sign_helper
			{
				inline static T call( const T& a )
				{
					return T( less_than( T( 0 ), a ) ) - T( less_than( a, T( 0 ) ) );
				}
			};

			template < typename T, typename U >
			struct sign_helper< T, U, false, false >
			{
				inline static T call( const T& a )
				{
					return T( greater_than( a, T( 0 ) ) );
				}
			};

			template < typename T, typename U >
			struct sign_helper< T, U, false, true >
			{
				inline static T call( const T& a )
				{
					typedef typename make_unsigned<T>::type UT;
					typedef typename make_unsigned<U>::type UU;
					const U shift( static_cast<U>( sizeof( U ) * 8 - 1 ) );
					const T y( UT( -a ) >> UU( shift ) );
					return ( a >> shift ) | y;
				}
			};

		}

		template < typename T, typename enable_if< is_arithmetic<T>::value >::type* = nullptr >
		inline T abs( T x )
		{
			return detail::abs_helper< T >::call( x );
		}

		template <>
		inline sint32 abs( sint32 x )
		{
			const sint32 y = x >> 31;
			return ( x ^ y ) - y;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T abs( const T& x )
		{
			return detail::unary_functor< T >::call( abs, x );
		}

		template < typename T, typename enable_if< is_arithmetic<T>::value >::type* = nullptr >
		inline T sign( T x )
		{
			return detail::sign_helper< T, T >::call( x );
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T sign( const T& x )
		{
			return detail::sign_helper< T >::call( x );
		}

		using ::floor;
		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T floor( const T& x )
		{
			return detail::unary_functor< T >::call( floor, x );
		}

		using ::trunc;
		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T trunc( const T& x )
		{
			return detail::unary_functor< T >::call( trunc, x );
		}

		using ::round;
		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T round( const T& x )
		{
			return detail::unary_functor< T >::call( round, x );
		}

		using ::ceil;
		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T ceil( const T& x )
		{
			return detail::unary_functor< T >::call( ceil, x );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T fract( T x )
		{
			return x - floor( x );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline T fract( const T& x )
		{
			return x - floor( x );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T mod( T a, T b )
		{
			return a - b * floor( a / b );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline T mod( const T& a, const T& b )
		{
			return a - b * floor( a / b );
		}

		template < typename T, typename enable_if< is_arithmetic<T>::value >::type* = nullptr >
		constexpr T min( T a, T b )
		{
			return a < b ? a : b;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T min( const T& a, value_type_t<T> b )
		{
			return detail::binary_scalar_functor< T >::call( min, a, b );
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T min( const T& a, const T& b )
		{
			return detail::binary_functor< T >::call( min, a, b );
		}

		template < typename T, typename enable_if< is_arithmetic<T>::value >::type* = nullptr >
		constexpr T max( T a, T b )
		{
			return a > b ? a : b;
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T max( const T& a, value_type_t<T> b )
		{
			return detail::binary_scalar_functor< T >::call( max, a, b );
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T max( const T& a, const T& b )
		{
			return detail::binary_functor< T >::call( max, a, b );
		}

		template < typename T, typename enable_if< is_arithmetic<T>::value >::type* = nullptr >
		constexpr T clamp( T x, T minv, T maxv )
		{
			return ::nv::math::min( ::nv::math::max( x, minv ), maxv );
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T clamp( const T& x, value_type_t<T> minv, value_type_t<T> maxv )
		{
			return ::nv::math::min( ::nv::math::max( x, minv ), maxv );
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T clamp( const T& x, const T& minv, const T& maxv )
		{
			return ::nv::math::min( ::nv::math::max( x, minv ), maxv );
		}

		template < typename T >
		constexpr T mix( T a, T b, bool m )
		{
			return m ? b : a;
		}

		template < typename T, typename enable_if< is_arithmetic<T>::value >::type* = nullptr >
		constexpr T mix( T a, T b, T m )
		{
			return detail::mix_helper< T, T, is_same< T, bool >::value >::call( a, b, m );
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T mix( const T& a, const T& b, value_type_t<T> m )
		{
			return detail::mix_helper< T, value_type_t<T>, is_same< value_type_t<T>, bool >::value >::call( a, b, m );
		}

		template < typename T, typename enable_if< is_vec<T>::value >::type* = nullptr >
		inline T mix( const T& a, const T& b, const T& m )
		{
			return detail::mix_helper< T >::call( a, b, m );
		}

	}
}

#endif // NV_STL_MATH_BASIC_HH
