// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file hash_map.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief hash_map
*/

#ifndef NV_STL_HASH_MAP_HH
#define NV_STL_HASH_MAP_HH

#include <nv/common.hh>
#include <nv/stl/container/hash_table.hh>
#include <nv/stl/container/hash_table_policy.hh>
#include <nv/stl/utility/pair.hh>

namespace nv
{

	template <
		typename Key,
		typename Query,
		typename Mapped,
		typename Hash,
		bool StoreHash,
		bool CompareWithHash
	>
	struct hash_map_entry_policy
	{
		typedef hash_table_standard_entry< Key, Mapped, Hash, StoreHash >        entry_type;
		typedef hash_table_standard_entry_hasher< entry_type, StoreHash >        hasher_type;
		typedef hash_table_standard_compare< entry_type, hasher_type, true, CompareWithHash > compare_type;
		typedef Query                            query_type;
		typedef typename entry_type::key_type    key_type;
		typedef typename entry_type::mapped_type mapped_type;
		typedef typename entry_type::hash_type   hash_type;
		typedef typename entry_type::value_type  value_type;

		template < typename T >
		static constexpr hash_type get_hash( const T& t )
		{
			return hasher_type::template get_hash< T >( t );
		}

		template < typename ComparableType >
		static constexpr bool entry_compare( const entry_type* entry, hash_type h, const ComparableType& key )
		{
			return compare_type::template apply< ComparableType >( entry, h, key );
		}
		static constexpr bool entry_compare( const entry_type* entry, hash_type h, const value_type& value )
		{
			return compare_type::template apply< key_type >( entry, h, entry_type::get_key( value ) );
		}

		template < typename... Args >
		static void entry_construct( entry_type* entry, hash_type hash_code, Args&&... params )
		{
			construct_object( entry_type::get_value( entry ), ::nv::forward<Args>( params )... );
			hasher_type::set_hash( entry, hash_code );
		}

		static void entry_destroy( entry_type* entry )
		{
			destroy_object( entry_type::get_value( entry ) );
		}

		static constexpr hash_type get_entry_hash( const entry_type* entry )
		{
			return hasher_type::get_hash( entry );
		}

		static constexpr hash_type get_value_hash( const value_type& value )
		{
			return hasher_type::template get_hash< key_type >( entry_type::get_key( value ) );
		}
	};
	
	template <
		typename Key,
		typename Mapped,
		typename Hash = size_t,
		typename EntryPolicy = hash_map_entry_policy< Key, Key, Mapped, Hash, false, false >,
		template < typename > class QueryConvertPolicy  = hash_table_no_extra_types_policy,
		template < typename > class InsertConvertPolicy = hash_table_no_extra_types_policy
	>
	class hash_map : public hash_table_base< EntryPolicy, QueryConvertPolicy >
	{
	public:
		typedef hash_table_base< EntryPolicy, QueryConvertPolicy >                base_type;
		typedef hash_map< Key, Mapped, Hash, EntryPolicy, QueryConvertPolicy > this_type;
		typedef typename base_type::value_type                                           value_type;
		typedef typename base_type::pointer                                              pointer;
		typedef typename base_type::const_pointer                                        const_pointer;
		typedef typename base_type::reference                                            reference;
		typedef typename base_type::const_reference                                      const_reference;
		typedef typename base_type::iterator                                             iterator;
		typedef typename base_type::const_iterator                                       const_iterator;
		typedef typename base_type::size_type                                            size_type;
		typedef typename base_type::difference_type                                      difference_type;
		typedef typename base_type::query_type                                           query_type;
		typedef typename base_type::key_type                                             key_type;
		typedef typename base_type::mapped_type                                          mapped_type;
		typedef typename base_type::node_type                                            node_type;
		typedef typename base_type::insert_return_type                                   insert_return_type;

		using base_type::base_type;

		mapped_type& operator[]( const key_type& key )
		{
			return ( *base_type::insert_key( key ).first ).second;
		}

		template < typename T >
		enable_if_t< InsertConvertPolicy<T>::value, mapped_type& >
		operator[]( const T& key )
		{
			return ( *base_type::insert_key( key ).first ).second;
		}

		mapped_type& at( const query_type& key )
		{
			iterator it = base_type::find( key );
			NV_ASSERT_ALWAYS( it != base_type::end(), "Key not found in map!" );
			return *it;
		}

		const mapped_type& at( const query_type& key ) const
		{
			const_iterator it = base_type::find( key );
			NV_ASSERT_ALWAYS( it != base_type::cend(), "Key not found in map!" );
			return *it;
		}

		template < typename T >
		enable_if_t< QueryConvertPolicy<T>::value, mapped_type& >
		at( const T& key )
		{
			iterator it = base_type::find( key );
			NV_ASSERT_ALWAYS( it != base_type::end(), "Key not found in map!" );
			return *it;
		}

		template < typename T >
		enable_if_t< QueryConvertPolicy<T>::value, const mapped_type& >
		at( const T& key ) const
		{
			const_iterator it = base_type::find( key );
			NV_ASSERT_ALWAYS( it != base_type::cend(), "Key not found in map!" );
			return *it;
		}

		template < typename M >
		insert_return_type assign( const key_type& k, M&& obj )
		{
			insert_return_type result = base_type::try_insert( k, nv::forward<M>( obj ) );
			if ( !result.second ) result.first->second = obj;
			return result;
		}

		template < typename M >
		insert_return_type assign( key_type&& k, M&& obj )
		{
			insert_return_type result = base_type::try_insert( nv::forward<key_type>( k ), nv::forward<M>( obj ) );
			if ( !result.second ) result.first->second = obj;
			return result;
		}

		template < typename T, typename M >
		enable_if_t< InsertConvertPolicy<T>::value, insert_return_type >
		assign( const T& k, M&& obj )
		{
			insert_return_type result = base_type::try_insert( k, nv::forward<M>( obj ) );
			if ( !result.second ) result.first->second = obj;
			return result;
		}

		template < typename T, typename M >
		enable_if_t< InsertConvertPolicy<T>::value, insert_return_type >
		assign( T&& k, M&& obj )
		{
			insert_return_type result = base_type::try_insert( nv::forward<T>( k ), nv::forward<M>( obj ) );
			if ( !result.second ) result.first->second = obj;
			return result;
		}

	};

}

#endif // NV_STL_HASH_MAP_HH
