// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_INTERFACE_MESH_DATA_HH
#define NV_INTERFACE_MESH_DATA_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/string.hh>
#include <nv/interface/data_channel.hh>
#include <nv/gfx/animation.hh>

namespace nv
{

	struct index_u16 { uint16 index; };
	struct index_u32 { uint32 index; };

	//using mesh_data = data_channel_set;

	// TODO : attribute/property implementation and read/write on every nmd::command

	class mesh_nodes_data
	{
		friend class mesh_creator;
		friend class mesh_nodes_creator;
	public:
		typedef vector< data_channel_set* > storage;
		typedef storage::const_iterator const_iterator;

		explicit mesh_nodes_data( shash64 name )
			: m_name( name ), m_frame_rate(0), m_duration(0.0f), m_flat( false )
		{
		}

		explicit mesh_nodes_data( shash64 name, uint16 a_fps, float a_frames, bool a_flat )
			: m_name( name ), m_frame_rate(a_fps), m_duration(a_frames), m_flat( a_flat )
		{
		}

		void push_back( data_channel_set* set )
		{
			m_data.push_back( set );
		}

		size_t size() const { return m_data.size(); }
		bool empty() const { return m_data.empty(); }

		bool is_animated( size_t i ) const
		{
			if ( i >= m_data.size() ) return false;
			return ( m_data[i]->size() > 0 );
		}

		const data_channel_set* get_by_hash( shash64 h ) const
		{
			for ( auto data : m_data )
			{
				if ( data->get_name() == h ) return data;
			}
			return nullptr;
		}

		int get_index_by_hash( shash64 h ) const
		{
			for ( uint32 i = 0; i < m_data.size(); ++i )
			{
				if ( m_data[ i ]->get_name() == h )
					return int( i );
			}
			return -1;
		}

		const data_channel_set* operator[]( size_t i ) const
		{
			return m_data[i];
		}

		const_iterator begin() const { return m_data.begin(); }
		const_iterator end() const { return m_data.end(); }

		bool is_flat() const { return m_flat; }
		uint16 get_frame_rate() const { return m_frame_rate; }
		float get_duration() const { return m_duration; }
		shash64 get_name() const { return m_name; }

		~mesh_nodes_data()
		{
			for ( auto data : m_data ) delete data;
		}

	private:
		vector< data_channel_set* > m_data;
		shash64 m_name;
		uint16  m_frame_rate;
		float   m_duration;
		bool    m_flat;
	};

	class mesh_data_pack
	{
		friend class mesh_creator;
	public:
		explicit mesh_data_pack( uint32 a_count, data_channel_set* a_meshes, mesh_nodes_data* a_nodes = nullptr )
		{
			m_count  = a_count;
			m_meshes = a_meshes;
			m_nodes  = a_nodes;
		}
		const data_channel_set* get_mesh( uint32 index ) const
		{
			if ( index >= m_count ) return nullptr;
			return &m_meshes[ index ];
		}
		const data_channel_set* get_mesh_by_hash( shash64 h ) const
		{
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_meshes[i].get_name() == h )
					return &m_meshes[i];
			}
			return nullptr;
		}
		uint32 get_count() const { return m_count; }
		const mesh_nodes_data* get_nodes() const { return m_nodes; }
		uint32 get_node_count() const { return m_nodes ? m_nodes->size() : 0; }
		~mesh_data_pack()
		{
			delete[] m_meshes;
			delete   m_nodes;
		}
	private:
		uint32            m_count;
		data_channel_set* m_meshes;
		mesh_nodes_data*  m_nodes;
	};
}

#endif // NV_INTERFACE_MESH_DATA_HH
