// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_BASE_COMMON_HH
#define NV_BASE_COMMON_HH

// NV Library version
#define NV_VERSION_MAJOR 0
#define NV_VERSION_MINOR 1
#define NV_VERSION_PATCH 0
#define NV_VERSION    (NV_VERSION_MAJOR << 16) | (NV_VERSION_MINOR << 8) | NV_VERSION_PATCH)

// Platform
#define NV_WINDOWS        1
#define NV_LINUX          2
#define NV_APPLE          3

// Compiler
#define NV_MSVC           1
#define NV_GNUC           2
#define NV_CLANG          3

// Endianess
#define NV_LITTLEENDIAN   1
#define NV_BIGENDIAN      2

// Bits
#define NV_32BIT          1
#define NV_64BIT          2

// Assumption
#define NV_ENDIANESS NV_LITTLEENDIAN

// Platform detection
#if defined( __WIN32__ ) || defined( _WIN32 )
#define NV_PLATFORM NV_WINDOWS
#elif defined( __APPLE_CC__)
#define NV_PLATFORM NV_APPLE
#else
#define NV_PLATFORM NV_LINUX
#endif

// Compiler detection
#if defined( _MSC_VER )
#define NV_COMPILER NV_MSVC
#define NV_COMP_VER _MSC_VER
#elif defined( __clang__ )
#define NV_COMPILER NV_CLANG
#define NV_COMP_VER (((__clang_major__)*100) + (__clang_minor__*10) + __clang_patchlevel__)
#elif defined( __GNUC__ )
#define NV_COMPILER NV_GNUC
#define NV_COMP_VER (((__GNUC__)*100) + (__GNUC_MINOR__*10) + __GNUC_PATCHLEVEL__)
#else
#error "Unknown compiler!"
#endif

// Architecture detection
#if defined(__x86_64__) || defined(_M_X64) || defined(__powerpc64__) || defined(__alpha__)
#define NV_ARCHITECTURE NV_64BIT
#elif defined(__ia64__) || defined(__s390__) || defined(__s390x__)
#define NV_ARCHITECTURE NV_64BIT
#else
#define NV_ARCHITECTURE NV_32BIT
#endif

// Platform specific settings.
#if NV_COMPILER == NV_MSVC
#ifdef _DEBUG
#define NV_DEBUG 1
#else
#define NV_DEBUG 0
#endif
#else
#ifdef NDEBUG
#define NV_DEBUG 0
#else
#define NV_DEBUG 1
#endif
#endif

#if NV_DEBUG
#ifndef NV_PROFILER
#define NV_PROFILER 
#endif
#endif

#ifdef NV_PROFILER
#undef NV_PROFILER
#define NV_PROFILER 1
#else
#define NV_PROFILER 0
#endif

#if NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#define NV_POSIX
#endif

#if NV_COMPILER == NV_MSVC && NV_COMP_VER < 1900
#error "MSVC 2015+ required!"
#endif

#if NV_COMPILER == NV_GNUC && NV_COMP_VER < 480
#error "GCC 4.8+ required!"
#endif

#if NV_COMPILER == NV_CLANG && NV_COMP_VER < 330
#error "clang 3.3+ required!"
#endif

#if NV_COMPILER == NV_MSVC 
#pragma warning( disable : 4201 ) // nonstandard extension used : nameless struct/union
#pragma warning( disable : 4510 ) // default constructor could not be generated
#pragma warning( disable : 4610 ) // object 'class' can never be instantiated - user-defined constructor required
#undef _SCL_SECURE_NO_WARNINGS // to prevent redefining
#define _SCL_SECURE_NO_WARNINGS
#endif

#define NV_STRINGIZE_DETAIL(x) #x
#define NV_STRINGIZE(x) NV_STRINGIZE_DETAIL(x)

#if NV_COMPILER == NV_MSVC
#define NV_WIDE_DETAIL(x) L ## x
#define NV_WIDE(x) NV_WIDE_DETAIL(x)
#endif

#if NV_COMPILER == NV_MSVC
#define NV_FUNC_SIG __FUNCSIG__
#define NV_DEPRECATED(func) __declspec(deprecated) func
#define NV_NORETURN __declspec(noreturn)
#define NV_NOALIAS __declspec(noalias)
#define NV_RESTRICT __declspec(restrict)
#define NV_RESTRICT_VAR __restrict
#define NV_MSVC_SUPRESS( warn_number ) __pragma( warning( suppress : warn_number ) )
#if NV_ARCHITECTURE == NV_64BIT
#define NV_OFFSET_OF(obj,m) (nv::size_t)( (nv::ptrdiff_t)&reinterpret_cast<const volatile char&>((((obj *)0)->m)) )
#else // NV_32BIT
#define NV_OFFSET_OF(obj,m) (nv::size_t)&reinterpret_cast<const volatile char&>((((obj *)0)->m))
#endif  
#elif NV_COMPILER == NV_GNUC || NV_COMPILER == NV_CLANG
#define NV_FUNC_SIG __PRETTY_FUNCTION__
#define NV_DEPRECATED(func) func __attribute__ ((deprecated))
#define NV_NORETURN __attribute__ ((__noreturn__))
#define NV_RESTRICT __restrict__
#define NV_RESTRICT_VAR __restrict__
#define NV_OFFSET_OF(obj, m) __builtin_offsetof(obj, m)
#define NV_MSVC_SUPRESS( warn_number ) 
#else
#error "Unknown compiler, cannot proceed!"
#endif 

#define NV_UNUSED(x) (void)(x)

// MSVC and GCC is too stupid to notice fully covered enums, clang 
// is picky about it
#if NV_COMPILER == NV_CLANG
#define NV_RETURN_COVERED_DEFAULT( value ) 
#else
#define NV_RETURN_COVERED_DEFAULT( value ) default : return value
#endif

#define NV_COUNT_OF(x) ((sizeof(x)/sizeof(0[x])) / (static_cast<nv::size_t>(!(sizeof(x) % sizeof(0[x])))))
#define NV_SAFE_ARRAY( arr, idx, def ) ( index < NV_COUNT_OF(arr) ? (arr)[idx] : (def) )

#define NV_TYPE_FAIL(T) (nv::static_assert_fail<T>::value)
#define NV_TEMPLATE_FAIL(T,message) static_assert( NV_TYPE_FAIL(T), message );

#ifdef min
#undef min
#endif
#ifdef max
#undef max
#endif
#ifdef isnan
#undef isnan
#endif
#ifdef isinf
#undef isinf
#endif
#ifdef log2
#undef log2
#endif

namespace nv
{

	template < typename T >
	struct static_assert_fail
	{
		static constexpr bool value = false;
	};

#if NV_COMPILER == NV_MSVC 
	using ::size_t;
#else
	using size_t    = decltype( sizeof(0) );
#endif
	using ptrdiff_t = decltype( static_cast<int*>(0) - static_cast<int*>(0) );
	using nullptr_t = decltype( nullptr );
	
#if NV_ARCHITECTURE == NV_64BIT
	using uintptr_t = unsigned long long;
#else
	using uintptr_t = unsigned int;
#endif

	// Typedefs for fixed size types.
	typedef signed   char      sint8;
	typedef signed   short     sint16;
	typedef signed   int       sint32;

	typedef unsigned char      uint8;
	typedef unsigned short     uint16;
	typedef unsigned int       uint32;

	typedef signed   long long sint64;
	typedef unsigned long long uint64;

	typedef unsigned char      uchar8;
	typedef unsigned short     uchar16;
	typedef unsigned int       uchar32;

	typedef signed char        schar8;
	typedef signed short       schar16;
	typedef signed long        schar32;

	typedef float              f32;
	typedef double             f64;

	typedef uint64             uid;

	struct empty_type {};

	template < typename T >
	class empty_base_class {};

	template < int a, int b, int c, int d >
	struct four_cc
	{
		static constexpr unsigned int value = (((((d << 8) | c) << 8) | b) << 8) | a;
	};

	class noncopyable
	{
	protected:
		constexpr noncopyable() = default;
		~noncopyable() = default;
		noncopyable( const noncopyable& ) = delete;
		noncopyable& operator=( const noncopyable& ) = delete;
	};

	template <typename OBJ, typename T>
	inline ptrdiff_t offset_of( T OBJ::*ptr )
	{
		return reinterpret_cast<ptrdiff_t>( &reinterpret_cast<const volatile char&>( ( static_cast<OBJ*>(0) )->*ptr ) );
	}

	template <typename T, typename U>
	constexpr T narrow_cast( const U& a )
	{
		return static_cast<T>( a & static_cast<T>( -1 ) );
	}

	template< typename T >
	void void_assign( T*& out, void* in )
	{
		out = reinterpret_cast<T*>( in );
	}

	enum log_level
	{
		LOG_NONE = 0,
		LOG_FATAL = 10,
		LOG_CRITICAL = 20,
		LOG_ERROR = 30,
		LOG_WARNING = 40,
		LOG_NOTICE = 50,
		LOG_INFO = 60,
		LOG_DEBUG = 80,
		LOG_TRACE = 100
	};

	using log_handler_t = void( *)( int, const char*, size_t );

	log_handler_t get_log_handler();
	void set_log_handler( log_handler_t );

} // namespace nv

static_assert( static_cast<nv::size_t>( -1 ) > 0, "size_t is signed!" );
static_assert( sizeof( nv::size_t ) >= 4, "size_t is smaller than 4 bytes!" );
static_assert( sizeof( nv::sint8 ) == 1, "sint8 size isn't 1 bytes" );
static_assert( sizeof( nv::sint16 ) == 2, "sint16 size isn't 2 bytes" );
static_assert( sizeof( nv::sint32 ) == 4, "sint32 size isn't 4 bytes" );
static_assert( sizeof( nv::sint64 ) == 8, "sint64 size isn't 8 bytes" );

static_assert( sizeof( nv::uint8 ) == 1, "uint8 size isn't 1 byte" );
static_assert( sizeof( nv::uint16 ) == 2, "uint16 size isn't 2 bytes" );
static_assert( sizeof( nv::uint32 ) == 4, "uint32 size isn't 4 bytes" );
static_assert( sizeof( nv::uint64 ) == 8, "uint64 size isn't 8 bytes" );

static_assert( sizeof( nv::f32 ) == 4, "float32 size isn't 4 bytes" );
static_assert( sizeof( nv::f64 ) == 8, "float64 size isn't 8 bytes" );

#include <nv/base/assert.hh>
#include <nv/base/rtti_support.hh>
#include <nv/base/new.hh>
#include <nv/base/capi.hh>

#endif // NV_BASE_COMMON_HH

