// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file position.hh
 * @author Kornel Kisielewicz
 * @brief UI 2D positioning classes
 */

#ifndef NV_CORE_POSITION_HH
#define NV_CORE_POSITION_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/range.hh>
#include <nv/stl/utility.hh>

namespace nv
{
	typedef ivec2 position;
	typedef ivec2 dimension;

	struct rectangle
	{
		typedef position::value_type value_type;
		typedef size_t size_type;
		typedef rectangle type;
		
		position ul;
		position lr;
		/**
		 * Creates a new rectangle assigned to {0, 0, 0, 0}.
		 */
		rectangle() : ul(), lr() {}
		
		/**
		 * Creates a new rectangle given a position.
		 *
		 * @param p The position to assign the rectangle to.
		 */
		explicit rectangle( position p ) : ul(p), lr(p) {}
		
		/**
		 * Creates a new rectangle given an upper-left and lower-right position.
		 *
		 * @param aul The position of the upper-left corner of the rectangle.
		 * @param alr The position of the lower-right corner of the rectangle.
		 */
		rectangle( position aul, position alr ) : ul(aul), lr(alr) {}
		
		/**
		 * Creates a new rectangle given an upper-left position, width, and height.
		 *
		 * @param aul The position of the upper-left corner of the rectangle.
		 * @param width The width of the rectangle.
		 * @param height The height of the rectangle.
		 */
		rectangle( position aul, value_type width, value_type height ) : ul(aul), lr(aul + position(width,height)) {}
        
        /**
         * Explicit Copy constructor
         */
        rectangle( const rectangle& r ) : ul( r.ul ), lr( r.lr ) {}
        
        /**
         * Explicit Copy assignment operator
         */
        rectangle& operator= (const rectangle& r) { ul = r.ul; lr = r.lr; return *this; }
		
		/**
		 * Sets the dimensions of the rectangle without moving the upper-left of the rectangle.
		 *
		 * @param d The new dimensions of the rectangle.
		 */
		rectangle& dim( dimension d ) { lr = ul + d; return *this; }

		/**
		 * Moves the rectangle to a new position while maintaining its size.
		 *
		 * @param p The new position of the rectangle's upper-left corner.
		 */
		rectangle& pos( position p )  { lr = p + (lr - ul); lr = p; return *this; }

		/**
		 * Sets the dimensions of the rectangle without moving the upper-left of the rectangle.
		 *
		 * @param d The new dimensions of the rectangle.
		 */
		void set_dimension( dimension d ) { lr = ul + d; }
		
		/**
		 * Moves the rectangle to a new position while maintaining its size.
		 *
		 * @param p The new position of the rectangle's upper-left corner.
		 */
		void set_position( position p ) { lr = p + (lr - ul); ul = p; }

		position ur() const { return position( lr.x, ul.y ); }
		position ll() const { return position( ul.x, lr.y ); }
		/**
		 * Gets the dimensions of the rectangle.  Synonym for get_size.
		 *
		 * @returns The dimensions of the rectangle.
		 * @see get_size
		 */
		dimension get_dimension() const { return lr - ul; }

		/**
		 * Gets the position of the upper-left corner of the rectangle.
		 *
		 * @returns The position of the rectangle's upper-left corner.
		 */
		position  get_position() const { return ul; }

		/**
		 * Gets the dimensions of the rectangle.  Synonym for get_dimension.
		 *
		 * @returns The dimensions of the rectangle.
		 * @see get_dimension
		 */
		dimension get_size() const { return lr - ul; }

		/**
		 * Gets the center of the rectangle.
		 *
		 * @returns The center of the rectangle.
		 */
		position get_center() const { return ( lr + ul ) / 2; }

		/**
		 * Gets the width of the rectangle.
		 *
		 * @returns The width of the rectangle.
		 */
		value_type get_width() const { return lr.x - ul.x; }

		/**
		 * Gets the height of the rectangle.
		 *
		 * @returns The height of the rectangle.
		 */
		value_type get_height() const { return lr.y - ul.y; }

		/**
		 * Gets the area of the rectangle.
		 *
		 * @returns The area of the rectangle.
		 */
		value_type get_area() const { return (lr.y - ul.y) * (lr.x - ul.x); }

		/**
		 * Gets the area of the rectangle.
		 *
		 * @returns The area of the rectangle, including the edge
		 */
		value_type get_enclosed_area() const { return (lr.y - ul.y + 1) * (lr.x - ul.x + 1); }

		/**
		 * Checks to see if the rectangle is backwards.
		 *
		 * @returns True if the rectangle's upper-left is above and left (or equal to) the rectangle's lower-right, false if it is not.
		 */
		bool is_valid() const {	return lr.x >= ul.x && lr.y >= ul.y; }

		/**
		 * Enlarges the rectangle by a given amount.  Each side is adjusted by the given amount, e.g. expanding by 1 increases the width and height by 2 each.
		 *
		 * @param value The amount to adjust each sides by.
		 */
		void expand( value_type value ) { position p(value,value); ul -= p; lr += p; }

		/**
		 * Reduces the rectangle by a given amount.  Each side is adjusted by the given amount, e.g. shrinking by 1 decreases the width and height by 2 each.
		 *
		 * @param value The amount to adjust each side by.
		 */
		void shrink( value_type value ) { position p(value,value); ul += p; lr -= p; }

		/**
		 * Gets a rectangle that is an expanded version of this rectangle.
		 *
		 * @param value The amount to adjust each side by.
		 * @returns An expanded rectangle.
		 * @see expand
		 */
		rectangle expanded( int value ) { position p(value,value); return rectangle(ul-p, lr+p); }

		/**
		 * Gets a rectangle that is a shrunk version of this rectangle.
		 *
		 * @param value The amount to adjust each side by.
		 * @returns A shrunk rectangle.
		 * @see shrink
		 */
		rectangle shrinked( int value ) { position p(value,value); return rectangle(ul+p, lr-p); }

		rectangle shrinked( dimension value ) { return rectangle(ul+value, lr-value); }

		/**
		 * Shifts the rectangle by a given amount.
		 *
		 * @param pos The amount to shift the rectangle by.
		 */
		rectangle& operator+=( const position& pos ) { ul += pos; lr += pos; return (*this); }

		/**
		 * Returns a rectangle shifted by a given amount.
		 *
		 * @param pos The amount to shift by. 
		 * @returns The shifted rectangle.
		 */
		rectangle operator+( const position& pos ) const {	rectangle r(*this); return r += pos; }

		/**
		 * Shifts the rectangle by a given amount.
		 *
		 * @param pos The amount to shift the rectangle by.
		 */
		rectangle& operator-=( const position& pos ) { ul -= pos; lr -= pos; return (*this); }

		/**
		 * Returns a rectangle shifted by a given amount.
		 *
		 * @param pos The amount to shift by.
		 * @returns The shifted rectangle.
		 */
		rectangle operator-( const position& pos ) const {	rectangle r(*this); return r -= pos; }

		/**
		 * Compares two rectangles to see if they are the same.
		 *
		 * @param r The rectangle to compare to.
		 * @returns True if the rectangles have the same positions and dimensions, false otherwise.
		 */
		bool operator==( const rectangle& r ) const { return r.ul == ul && r.lr == lr; }

		/**
		 * Compares two rectangles to see if they are different.
		 *
		 * @param r The rectangle to compare to.
		 * @returns True if the rectangles have different positions or dimensions, false otherwise.
		 */
		bool operator!=( const rectangle& r ) const { return r.ul != ul || r.lr != lr; }

		/**
		 * Checks if a position is within the bounds of this rectangle.
		 *
		 * @param r The position to check.
		 * @returns True if the position is inside or on the edge of the rectangle, false otherwise.
		 */
		bool contains( const position& r ) const{ return lr.y >= r.y && ul.y <= r.y && lr.x >= r.x && ul.x <= r.x; }

		/**
		 * Checks if a rectangle is within the bounds of this rectangle.
		 *
		 * @param r The rectangle to check.
		 * @returns True if the entire rectangle to check is inside or on the edge of this rectangle, false otherwise.
		 */
		bool contains( const rectangle& r ) const { return contains( r.ul ) && contains( r.lr ); }

		/**
		 * Checks if another rectangle overlaps this one.
		 *
		 * @param r The rectangle to check.
		 * @returns True if any part of the rectangle to check overlaps this rectangle, false otherwise.
		 */
		bool collides( const rectangle& r ) const { return lr.y > r.ul.y && ul.y < r.lr.y && lr.x > r.ul.x && ul.x < r.lr.x; }

		/**
		 * Limits the region of the rectangle to the inidicated rectangle.
		 *
		 * @param r The rectangle that this rectangle is confined to.
		 * @returns True if the rectangle was changed, false otherwise.
		 */
		bool clamp_to( const rectangle& r )
		{
			if (r.contains(*this)) return false;
			ul = math::max( ul, r.ul );
			lr = math::min( lr, r.lr );
			ul = math::min( ul, lr );
			return true;
		}

		/**
		 * Limits the size of the rectangle to the given rectangle's size.
		 *
		 * @param r The rectangle representing the maximum dimensions this rectangle can be.
		 * @returns True if the rectangle needed to be resized, false otherwise.
		 */
		bool constrain_to( const rectangle& r )
		{
			if ( r.get_width() < get_width() || r.get_height() < get_height() ) return false;
			( *this ) += math::min( r.ul - ul, position() );
			( *this ) -= math::min( lr - r.lr, position() );
			return true;
		}

		/**
		 * Fixes an invalid rectangle.
		 */
		void repair() 
		{
			if (ul.x > lr.x) swap( ul.x, lr.x );
			if (ul.y > lr.y) swap( ul.y, lr.y );
		}

		/**
		 * Expands the rectangle to just include the given point.
		 *
		 * @param p The point to include in the rectangle.
		 */
		void include_point( position p )
		{
			lr = math::max( lr, p );
			ul = math::min( ul, p );
		}

	};

}

#endif // NV_CORE_POSITION_HH
