// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file common.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief math common
*/

#ifndef NV_STL_MATH_COMMON_HH
#define NV_STL_MATH_COMMON_HH

#include <nv/common.hh>
#include <nv/stl/utility/common.hh>
#include <nv/stl/type_traits.hh>
#if NV_COMPILER == NV_GNUC
#pragma GCC system_header
#elif NV_COMPILER == NV_CLANG
#pragma clang system_header
#endif

#include <nv/base/cmath.hh>
#include <glm/glm.hpp>
#include <glm/gtc/quaternion.hpp>
#include <glm/gtx/vector_angle.hpp>

namespace nv
{

	namespace math
	{

		// TODO: Remove
		using glm::inverse;
		using glm::transpose;

		using glm::detail::component_count_t;
		using glm::detail::component_count;

		using glm::ctor;

		using ::cos;
		using ::sin;
		using ::acos;
		using ::asin;
		using ::cosh;
		using ::sinh;
		using ::tan;

		using ::sqrt;
		using ::pow;
		using ::log2;
		using ::log;
		using ::exp;

		using ::floor;
		using ::round;

#if 0
		template < typename T > using tvec2 = ::glm::detail::tvec2<T, glm::precision::highp>;
		template < typename T > using tvec3 = ::glm::detail::tvec3<T, glm::precision::highp>;
		template < typename T > using tvec4 = ::glm::detail::tvec4<T, glm::precision::highp>;
		template < typename T > using tmat2 = ::glm::detail::tmat2x2<T, glm::precision::highp>;
		template < typename T > using tmat3 = ::glm::detail::tmat3x3<T, glm::precision::highp>;
		template < typename T > using tmat4 = ::glm::detail::tmat4x4<T, glm::precision::highp>;
		template < typename T > using tquat = ::glm::detail::tmat4x4<T, glm::precision::highp>;
#else
		template < typename T > using tvec2 = ::glm::tvec2<T>;
		template < typename T > using tvec3 = ::glm::tvec3<T>;
		template < typename T > using tvec4 = ::glm::tvec4<T>;
		template < typename T > using tmat2 = ::glm::tmat2x2<T>;
		template < typename T > using tmat3 = ::glm::tmat3x3<T>;
		template < typename T > using tmat4 = ::glm::tmat4x4<T>;
		template < typename T > using tquat = ::glm::tquat<T, glm::highp>;
#endif

		typedef glm::quat quat;

		template < typename T > struct is_vec : false_type {};
		template < typename T > struct is_mat : false_type {};
		template < typename T > struct is_quat : false_type {};

		template < typename T > struct is_fp : is_floating_point< T > {};
		template < typename T > struct is_fp_vec : false_type {};
		template < typename T > struct is_fp_mat : false_type {};
		template < typename T > struct is_fp_quat : false_type {};

		template < typename T > struct is_bool_vec : false_type {};

		template < typename T > struct is_vec < tvec2< T > > : true_type {};
		template < typename T > struct is_vec < tvec3< T > > : true_type {};
		template < typename T > struct is_vec < tvec4< T > > : true_type {};
		template < typename T > struct is_mat < tmat2< T > > : true_type {};
		template < typename T > struct is_mat < tmat3< T > > : true_type {};
		template < typename T > struct is_mat < tmat4< T > > : true_type {};
		template < typename T > struct is_quat< tquat< T > > : true_type {};

		template < typename T > struct is_fp     < tvec2< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp     < tvec3< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp     < tvec4< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp_vec < tvec2< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp_vec < tvec3< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp_vec < tvec4< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp_mat < tmat2< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp_mat < tmat3< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp_mat < tmat4< T > > : is_floating_point< T > {};
		template < typename T > struct is_fp_quat< tquat< T > > : is_floating_point< T > {};

		template <> struct is_bool_vec < tvec2< bool > > : true_type {};
		template <> struct is_bool_vec < tvec3< bool > > : true_type {};
		template <> struct is_bool_vec < tvec4< bool > > : true_type {};

		namespace detail
		{
			template < typename Vec, typename R = value_type_t<Vec>, typename T = value_type_t<Vec> >
			struct unary_functor {};

			template < typename R, typename T >
			struct unary_functor< tvec2< T >, R, T >
			{
				inline static tvec2<R> call( R( *func )( T ), const tvec2<T>& v )
				{
					return tvec2<R>( func( v.x ), func( v.y ) );
				}
			};

			template < typename R, typename T >
			struct unary_functor< tvec3< T >, R, T >
			{
				inline static tvec3<R> call( R( *func )( T ), const tvec3<T>& v )
				{
					return tvec3<R>( func( v.x ), func( v.y ), func( v.z ) );
				}
			};

			template < typename R, typename T >
			struct unary_functor< tvec4< T >, R, T >
			{
				inline static tvec4<R> call( R( *Func ) ( T x ), const tvec4<T>& v )
				{
					return tvec4<R>( func( v.x ), func( v.y ), func( v.z ), func( v.w ) );
				}
			};

			template < typename Vec, typename T = value_type_t<Vec> >
			struct binary_functor {};

			template < typename T >
			struct binary_functor< tvec2< T >, T >
			{
				inline static tvec2<T> call( T( *func ) ( T, T ), const tvec2<T>& a, const tvec2<T>& b )
				{
					return tvec2<T>( func( a.x, b.x ), func( a.y, b.y ) );
				}
			};

			template < typename T >
			struct binary_functor< tvec3< T >, T >
			{
				inline static tvec3<T> call( T( *func ) ( T, T ), const tvec3<T>& a, const tvec3<T>& b )
				{
					return tvec3<T>( func( a.x, b.x ), func( a.y, b.y ), func( a.z, b.z ) );
				}
			};

			template < typename T >
			struct binary_functor< tvec4< T >, T >
			{
				inline static tvec4<T> call( T( *func ) ( T, T ), const tvec4<T>& a, const tvec4<T>& b )
				{
					return tvec4<T>( func( a.x, b.x ), func( a.y, b.y ), func( a.z, b.z ), func( a.w, b.w ) );
				}
			};

			template < typename Vec, typename T = value_type_t<Vec> >
			struct binary_scalar_functor {};

			template < typename T >
			struct binary_scalar_functor< tvec2< T >, T >
			{
				inline static tvec2<T> call( T( *func ) ( T, T ), const tvec2<T>& a, T b )
				{
					return tvec2<T>( func( a.x, b ), func( a.y, b ) );
				}
			};

			template < typename T >
			struct binary_scalar_functor< tvec3< T >, T >
			{
				inline static tvec3<T> call( T( *func ) ( T, T ), const tvec3<T>& a, T b )
				{
					return tvec3<T>( func( a.x, b ), func( a.y, b ), func( a.z, b ) );
				}
			};

			template < typename T >
			struct binary_scalar_functor< tvec4< T >, T >
			{
				inline static tvec4<T> call( T( *func ) ( T, T ), const tvec4<T>& a, T b )
				{
					return tvec4<T>( func( a.x, b ), func( a.y, b ), func( a.z, b ), func( a.w, b ) );
				}
			};
		}
	}

	typedef math::tvec2<sint8> i8vec2;
	typedef math::tvec3<sint8> i8vec3;
	typedef math::tvec4<sint8> i8vec4;

	typedef math::tvec2<sint16> i16vec2;
	typedef math::tvec3<sint16> i16vec3;
	typedef math::tvec4<sint16> i16vec4;

	typedef math::tvec2<uint8> u8vec2;
	typedef math::tvec3<uint8> u8vec3;
	typedef math::tvec4<uint8> u8vec4;

	typedef math::tvec2< float > vec2;
	typedef math::tvec3< float > vec3;
	typedef math::tvec4< float > vec4;

	typedef math::tvec2< int > ivec2;
	typedef math::tvec3< int > ivec3;
	typedef math::tvec4< int > ivec4;

	typedef math::tmat2< float > mat2;
	typedef math::tmat3< float > mat3;
	typedef math::tmat4< float > mat4;

	typedef math::quat quat;



	template < typename T > 
	struct make_unsigned< math::tvec2< T > >
	{
		typedef math::tvec2< make_unsigned_t< T > > type;
	};

	template < typename T >
	struct make_unsigned< math::tvec3< T > >
	{
		typedef math::tvec3< make_unsigned_t< T > > type;
	};

	template < typename T >
	struct make_unsigned< math::tvec4< T > >
	{
		typedef math::tvec4< make_unsigned_t< T > > type;
	};

	template < typename T >
	struct make_signed< math::tvec2< T > >
	{
		typedef math::tvec2< make_signed_t< T > > type;
	};

	template < typename T >
	struct make_signed< math::tvec3< T > >
	{
		typedef math::tvec3< make_signed_t< T > > type;
	};

	template < typename T >
	struct make_signed< math::tvec4< T > >
	{
		typedef math::tvec4< make_signed_t< T > > type;
	};

}

#endif // NV_STL_MATH_COMMON_HH
