// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file angle.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief angle related math
*/

#ifndef NV_STL_MATH_ANGLE_HH
#define NV_STL_MATH_ANGLE_HH

#include <nv/stl/math/common.hh>
#include <nv/stl/math/geometric.hh>
#include <nv/stl/math/matrix_transform.hh>

namespace nv
{
	
	namespace math
	{

		template < typename T >
		inline tvec3<T> slerp( const tvec3<T>& a, const tvec3<T>& b, const T& v )
		{
			T angle     = acos( dot( a, b ) );
			T sin_angle = sin( angle );
			T t1 = sin( ( static_cast<T>( 1 ) - v ) * angle ) / sin_angle;
			T t2 = sin( v * angle ) / sin_angle;
			return a * t1 + b * t2;
		}

		template < typename T >
		inline tvec2<T> rotate( const tvec2<T>& v, const T& angle )
		{
			tvec2<T> result;
			T cosa( cos( angle ) );
			T sina( sin( angle ) );
			result.x = v.x * cosa - v.y * sina;
			result.y = v.x * sina + v.y * cosa;
			return result;
		}

		template < typename T >
		inline tvec3<T> rotate( const tvec3<T>& v, const T& angle, const tvec3<T>& normal )
		{
			return tmat3<T>( rotate( angle, normal ) ) * v;
		}

		template < typename T >
		inline tvec4<T> rotate( const tvec4<T>& v, const T& angle, const tvec3<T>& normal )
		{
			return math::rotate( angle, normal ) * v;
		}

		template < typename T >
		inline tvec3<T> rotate_x( const tvec3<T>& v, const T& angle )
		{
			tvec3<T> result( v );
			T cosa( cos( angle ) );
			T sina( sin( angle ) );

			result.y = v.y * cosa - v.z * sina;
			result.z = v.y * sina + v.z * cosa;
			return result;
		}

		template < typename T >
		inline tvec3<T> rotate_y( const tvec3<T>& v, const T& angle )
		{
			tvec3<T> result = v;
			T cosa( cos( angle ) );
			T sina( sin( angle ) );

			result.x =  v.x * cosa + v.z * sina;
			result.z = -v.x * sina + v.z * cosa;
			return result;
		}

		template < typename T >
		inline tvec3<T> rotate_z( const tvec3<T>& v, const T& angle )
		{
			tvec3<T> result = v;
			T cosa( cos( angle ) );
			T sina( sin( angle ) );

			result.x = v.x * cosa - v.y * sina;
			result.y = v.x * sina + v.y * cosa;
			return result;
		}

		template < typename T >
		inline tvec4<T> rotate_x( const tvec4<T>& v, const T& angle )
		{
			tvec3<T> result = v;
			T cosa( cos( angle ) );
			T sina( sin( angle ) );

			result.y = v.y * cosa - v.z * sina;
			result.z = v.y * sina + v.z * cosa;
			return result;
		}

		template < typename T >
		inline tvec4<T> rotate_y( const tvec4<T>& v, const T& angle )
		{
			tvec3<T> result = v;
			T cosa( cos( angle ) );
			T sina( sin( angle ) );

			result.x =  v.x * cosa + v.z * sina;
			result.z = -v.x * sina + v.z * cosa;
			return result;
		}

		template < typename T >
		inline tvec4<T> rotate_z( const tvec4<T>& v, const T& angle )
		{
			tvec3<T> result = v;
			T cosa( cos( angle ) );
			T sina( sin( angle ) );

			result.x = v.x * cosa - v.y * sina;
			result.y = v.x * sina + v.y * cosa;
			return result;
		}

		template < typename T >
		inline tmat4<T> orientation( const tvec3<T>& normal, const tvec3<T>& up )
		{
			if ( all( equal( normal, up ) ) ) return tmat4<T>( T( 1 ) );
			tvec3<T> raxis = cross( up, normal );
			T rangle = acos( dot( normal, up ) );
			return rotate( rangle, raxis );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T angle(	T a, T y )
		{
			return acos( clamp( dot( a, b ), T( -1 ), T( 1 ) ) );
		}

		template < typename T, typename enable_if< is_fp_vec<T>::value >::type* = nullptr >
		inline value_type_t<T> angle( const T& a, const T& b )
		{
			return acos( clamp( dot( a, b ), T( -1 ), T( 1 ) ) );
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T oriented_angle( const tvec2<T>& a, const tvec2<T>& b )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			const T result( acos( clamp( dot( a, b ), T( -1 ), T( 1 ) ) ) );

			if ( all( epsilon_equal( y, math::rotate( a, result ), T( 0.0001 ) ) ) )
				return result;
			else
				return -result;
		}

		template < typename T, typename enable_if< is_floating_point<T>::value >::type* = nullptr >
		inline T oriented_angle( const tvec3<T>& a, const tvec3<T>& b, const tvec3<T>& ref )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );

			const T result( acos( clamp( dot( a, b ), T( -1 ), T( 1 ) ) ) );
			return mix( result, -result, dot( ref, cross( a, b ) ) < T( 0 ) );
		}

	}

}

#endif // NV_STL_MATH_ANGLE_HH
