// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file camera.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Camera interface class
 */

#ifndef NV_INTERFACE_CAMERA_HH
#define NV_INTERFACE_CAMERA_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>

namespace nv
{
	class camera
	{
	public:
		camera() {}

		void set_lookat( const vec3& eye, const vec3& focus, const vec3& up )
		{
			m_view      = math::look_at( eye, focus, up );
			m_position  = eye;
			m_direction = math::normalize( focus - eye );
		}

		void set_perspective( f32 fov, f32 aspect, f32 near, f32 far )
		{
			m_projection = math::perspective( math::radians( fov ), aspect, near, far );
		}
		void set_ortho( f32 left, f32 right, f32 bottom, f32 top, f32 near = -1.0f, f32 far = 1.0f )
		{
			m_projection = math::ortho( left, right, bottom, top, near, far );
		}
		const mat4& get_projection() const 
		{
			return m_projection;
		}
		const mat4& get_view() const 
		{
			return m_view;
		}
		const vec3& get_position() const
		{
			return m_position;
		}
		const vec3& get_direction() const
		{
			return m_direction;
		}
	private:
		bool m_dirty;
		vec3 m_position;
		vec3 m_direction;
		mat4 m_projection;
		mat4 m_view;
	};

	// TODO - this and camera should have dirty states
	class scene_state
	{
	public:
		const camera& get_camera() const   { return m_camera; }
		camera& get_camera()               { return m_camera; }
		void set_camera( const camera& c ) { m_camera = c; }
		void set_model( const mat4& m )    { m_model  = m; } 
		void set_viewport( const ivec4& v ){ m_viewport = v; }
		const ivec4& get_viewport()  const { return m_viewport; }
		const mat4& get_model()      const { return m_model; }
		const mat4& get_view()       const { return m_camera.get_view(); }
		const mat4& get_projection() const { return m_camera.get_projection(); }
		mat4 get_modelview()  const { return get_view() * m_model; }
		mat4 get_mvp()        const { return m_camera.get_projection() * get_modelview(); }

		mat4 get_view_inv()   const { return math::inverse( get_view() ); }
		mat4 get_model_inv()  const { return math::inverse( get_model() ); }
		mat3 get_normal()     const { return math::transpose( math::inverse( mat3( get_modelview() ) ) ); }
	protected:
		mat4   m_model;
		camera m_camera; 
		ivec4  m_viewport;
	};


}

#endif // NV_INTERFACE_CAMERA_HH
