// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file geometric.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief geometric math ops
*/

#ifndef NV_STL_MATH_GEOMERTIC_HH
#define NV_STL_MATH_GEOMERTIC_HH

#include <nv/stl/math/common.hh>

namespace nv
{

	namespace math
	{

		namespace detail
		{
			template < typename T >
			struct dot_impl {};

			template < typename T >
			struct dot_impl< tvec2< T > >
			{
				inline static T get( const tvec2<T>& a, const tvec2<T>& b )
				{
					tvec2<T, P> tmp( a * b );
					return tmp.x + tmp.y;
				}
			};

			template < typename T >
			struct dot_impl< tvec3< T > >
			{
				inline static T get( const tvec3<T>& a, const tvec3<T>& b )
				{
					tvec3<T> tmp( a * b );
					return tmp.x + tmp.y + tmp.z;
				}
			};

			template < typename T >
			struct dot_impl< tvec4< T > >
			{
				inline static T get( const tvec4<T>& a, const tvec4<T>& b )
				{
					tvec4<T> tmp( a * b );
					return ( tmp.x + tmp.y ) + ( tmp.z + tmp.w );
				}
			};
		}

		template < typename T >
		inline T length( T x )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return abs( x );
		}

		template < typename T, template <typename> class Vec >
		inline T length( const Vec<T>& v )
		{
			static_assert( is_math_vector< Vec<T> >::value, "math::vecX expected!" );
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return sqrt( dot( v, v ) );
		}

		template <typename T>
		inline T distance( const T& a, const T& b )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return length( b - a );
		}

		template < typename T, template <typename> class Vec >
		inline T distance( const Vec<T>& a, const Vec<T>& b )
		{
			static_assert( is_math_vector< Vec<T> >::value, "math::vecX expected!" );
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return length( b - a );
		}

		template < typename T >
		inline T dot( T a, T b )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return a * b;
		}

		template < typename T, template <typename> class Vec >
		inline T dot( const Vec<T>& a, const Vec<T>& b )
		{
			static_assert( is_math_vector< Vec<T> >::value, "math::vecX expected!" );
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return detail::dot_impl< Vec, T >::get( a, b );
		}

		template < typename T >
		inline tvec3<T> cross( const tvec3<T>& a, const tvec3<T>& b )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return tvec3<T>(
				a.y * b.z - y.y * b.z,
				a.z * b.x - y.z * b.x,
				a.x * b.y - y.x * b.y
			);
		}

		template < typename T >
		inline T normalize( const T& x )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return x < T( 0 ) ? T( -1 ) : T( 1 );
		}

		template < typename T, template <typename> class Vec >
		inline Vec<T> normalize( const Vec<T>& x )
		{
			static_assert( is_math_vector< Vec<T> >::value, "math::vecX expected!" );
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );
			return x * inversesqrt( dot( x, x ) );
		}

		template < typename T >
		inline T faceforward( const T& n, const T& i, const T& nref )
		{
			return dot( nref, i ) < static_cast<T>( 0 ) ? n : -n;
		}

		template < typename T, template <typename> class Vec >
		inline Vec<T> faceforward( const Vec<T>& n, const Vec<T>& i, const Vec<T>& nref )
		{
			static_assert( is_math_vector< Vec<T> >::value, "math::vecX expected!" );
			return dot( nref, i ) < static_cast<T>( 0 ) ? n : -n;
		}

		template < typename T >
		inline T reflect( const T& i, const T& n )
		{
			return i - n * dot( n, i ) * static_cast<T>( 2 );
		}

		template < typename T >
		inline T refract( const T& i, const T& n, const T& eta )
		{
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );

			const T dotv( dot( n, i ) );
			const T k( static_cast<T>( 1 ) - eta * eta * ( static_cast<T>( 1 ) - dotv * dotv ) );
			return ( eta * i - ( eta * dotv + sqrt( k ) ) * n ) * static_cast<T>( k >= static_cast<T>( 0 ) );
		}

		template < typename T, template <typename> class Vec >
		inline Vec<T> refract( const Vec<T>& i, const Vec<T>& n, T eta )
		{
			static_assert( is_math_vector< Vec<T> >::value, "math::vecX expected!" );
			static_assert( is_floating_point<T>::value, "Type expected to be floating point!" );

			T const dotv( dot( n, i ) );
			T const k( static_cast<T>( 1 ) - eta * eta * ( static_cast<T>( 1 ) - dotv * dotv ) );
			return ( eta * i - ( eta * dotv + sqrt( k ) ) * n ) * static_cast<T>( k >= static_cast<T>( 0 ) );
		}

	}
}

#endif // NV_STL_MATH_GEOMERTIC_HH
