// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file iterator.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief STL iterator library
*/

#ifndef NV_STL_ITERATOR_HH
#define NV_STL_ITERATOR_HH

#include <nv/common.hh>
#include <nv/stl/type_traits/common.hh>

namespace nv
{
	struct input_iterator_tag {};
	struct output_iterator_tag {};
	struct forward_iterator_tag : input_iterator_tag {};
	struct bidirectional_iterator_tag : forward_iterator_tag {};
	struct random_access_iterator_tag : bidirectional_iterator_tag {};
	struct block_access_iterator_tag : random_access_iterator_tag{};

	template < typename Category, typename T, typename Distance = ptrdiff_t,
		typename Pointer = T*, typename Reference = T& >
	struct iterator
	{
		typedef T         value_type;
		typedef Distance  difference_type;
		typedef Pointer   pointer;
		typedef Reference reference;
		typedef Category  iterator_category;
	};

	template < typename Iterator >
	struct iterator_traits
	{
		typedef typename Iterator::iterator_category iterator_category;
		typedef typename Iterator::value_type        value_type;
		typedef typename Iterator::difference_type   difference_type;
		typedef typename Iterator::pointer           pointer;
		typedef typename Iterator::reference         reference;
	};

	template < typename T >
	struct iterator_traits < T* >
	{
		typedef block_access_iterator_tag iterator_category;
		typedef T                         value_type;
		typedef ptrdiff_t                 difference_type;
		typedef T*                        pointer;
		typedef T&                        reference;
	};

	template < typename T >
	struct iterator_traits < const T* >
	{
		typedef block_access_iterator_tag iterator_category;
		typedef T                         value_type;
		typedef ptrdiff_t                 difference_type;
		typedef const T*                  pointer;
		typedef const T&                  reference;
	};

	template <>
	struct iterator_traits < void* >
	{
		typedef block_access_iterator_tag iterator_category;
		typedef uint8                     value_type;
		typedef ptrdiff_t                 difference_type;
		typedef void*                     pointer;
		typedef uint8&                    reference;
	};

	template <>
	struct iterator_traits < const void* >
	{
		typedef block_access_iterator_tag iterator_category;
		typedef uint8                     value_type;
		typedef ptrdiff_t                 difference_type;
		typedef const void*               pointer;
		typedef const uint8&              reference;
	};

	template <typename Iterator>
	class reverse_iterator : public iterator 
	  < typename iterator_traits<Iterator>::iterator_category,
		typename iterator_traits<Iterator>::value_type,
		typename iterator_traits<Iterator>::difference_type,
		typename iterator_traits<Iterator>::pointer,
		typename iterator_traits<Iterator>::reference >
	{
	public:
		typedef Iterator                                              iterator_type;
		typedef typename iterator_traits<Iterator>::iterator_category iterator_category;
		typedef typename iterator_traits<Iterator>::value_type        value_type;
		typedef typename iterator_traits<Iterator>::pointer           pointer;
		typedef typename iterator_traits<Iterator>::reference         reference;
		typedef typename iterator_traits<Iterator>::difference_type   difference_type;

	public:
		constexpr reverse_iterator() : m_base() {}
		constexpr explicit reverse_iterator( iterator_type i ) : m_base( i ) {}

		template <typename U>
		constexpr reverse_iterator( const reverse_iterator<U>& ri ) : m_base( ri.base() ) {}

		constexpr iterator_type base() const { return m_base; }
		inline reference operator*( ) const { iterator_type i( m_base ); return *--i; }
		inline pointer operator->( ) const { return &( operator*( ) ); }
		inline reverse_iterator& operator++( ) { --m_base; return *this; }
		inline reverse_iterator operator++( int ) { reverse_iterator ri( *this ); --m_base; return ri; }
		inline reverse_iterator& operator--( ) { ++m_base; return *this; }
		inline reverse_iterator operator--( int ) { reverse_iterator ri( *this ); ++m_base; return ri; }
		inline reverse_iterator operator+( difference_type n ) const { return reverse_iterator( m_base - n ); }
		inline reverse_iterator& operator+=( difference_type n ) { m_base -= n; return *this; }
		inline reverse_iterator operator-( difference_type n ) const { return reverse_iterator( m_base + n ); }
		inline reverse_iterator& operator-=( difference_type n ) { m_base += n; return *this; }
		constexpr reference operator[]( difference_type n ) const { return m_base[-n - 1]; }
	protected:
		iterator_type m_base;
	};

	template < typename Iterator1, typename Iterator2 >
	constexpr bool operator==( const reverse_iterator< Iterator1 >& lhs, const reverse_iterator< Iterator2 >& rhs )
	{ return lhs.base() == rhs.base(); }
	template < typename Iterator1, typename Iterator2 >
	constexpr bool operator!=( const reverse_iterator< Iterator1 >& lhs, const reverse_iterator< Iterator2 >& rhs )
	{ return lhs.base() != rhs.base(); }
	template < typename Iterator1, typename Iterator2 >
	constexpr bool operator<( const reverse_iterator< Iterator1 >& lhs, const reverse_iterator< Iterator2 >& rhs )
	{ return lhs.base() > rhs.base(); }
	template < typename Iterator1, typename Iterator2 >
	constexpr bool operator>( const reverse_iterator< Iterator1>& lhs, const reverse_iterator< Iterator2 >& rhs )
	{ return lhs.base() < rhs.base(); }
	template < typename Iterator1, typename Iterator2 >
	constexpr bool operator<=( const reverse_iterator< Iterator1 >& lhs, const reverse_iterator<Iterator2 >& rhs )
	{ return lhs.base() >= rhs.base(); }
	template < typename Iterator1, typename Iterator2 >
	constexpr bool operator>=( const reverse_iterator< Iterator1 >& lhs, const reverse_iterator<Iterator2 >& rhs )
	{ return lhs.base() <= rhs.base(); }
	template < typename Iterator1, typename Iterator2 >
	constexpr typename reverse_iterator<Iterator1>::difference_type operator-( const reverse_iterator<Iterator1>& lhs, const reverse_iterator<Iterator2>& rhs )
	{ return rhs.base() - lhs.base(); }
	template < typename Iterator > constexpr reverse_iterator<Iterator> operator+( typename reverse_iterator<Iterator>::difference_type n, const reverse_iterator<Iterator>& iter )
	{ return reverse_iterator<Iterator>( iter.base() - n ); }

	namespace detail
	{
		template < typename InputIterator >
		inline typename iterator_traits< InputIterator >::difference_type
		distance_impl( InputIterator first, InputIterator last, input_iterator_tag )
		{
			typename iterator_traits< InputIterator >::difference_type n = 0;
			while ( first != last )
			{
				++first;
				++n;
			}
			return n;
		}

		template < typename RandomAccessIterator >
		inline typename iterator_traits< RandomAccessIterator >::difference_type
		distance_impl( RandomAccessIterator first, RandomAccessIterator last, random_access_iterator_tag )
		{
			return last - first;
		}

		template < typename InputIterator, typename Distance >
		inline void advance_impl( InputIterator& iter, Distance n, input_iterator_tag )
		{
			while ( n-- ) ++iter;
		}

		template < typename BidirectionalIterator, typename Distance >
		inline void advance_impl( BidirectionalIterator& iter, Distance n, bidirectional_iterator_tag )
		{
			if ( n > 0 )
			{
				while ( n-- ) ++iter;
			}
			else
			{
				while ( n++ ) --iter;
			}
		}

		template < typename RandomAccessIterator, typename Distance >
		inline void advance_impl( RandomAccessIterator& iter, Distance n, random_access_iterator_tag )
		{
			iter += n;
		}

	}

	template < typename InputIterator >
	inline typename iterator_traits< InputIterator >::difference_type
		distance( InputIterator first, InputIterator last )
	{
		typedef typename iterator_traits< InputIterator >::iterator_category category;
		return detail::distance_impl( first, last, category() );
	}

	// Returns 0 for input iterators, distance for others
	// this is an extension to the standard
	template < typename InputIterator >
	inline typename iterator_traits< InputIterator >::difference_type
		estimate_distance( InputIterator first, InputIterator last )
	{
		typedef typename iterator_traits< InputIterator >::iterator_category category;
		return !is_same< category, input_iterator_tag >::value ? detail::distance_impl( first, last, category() ) : 0;
	}


	template < typename InputIterator, typename Distance >
	inline void advance( InputIterator& iter, Distance n )
	{
		typedef typename iterator_traits< InputIterator >::iterator_category category;
		detail::advance_impl( iter, n, category() );
	}

}

#endif // NV_STL_ITERATOR_HH
