// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file hash_table_policy.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief hash table policy classes
*/

#ifndef NV_STL_HASH_TABLE_POLICY_HH
#define NV_STL_HASH_TABLE_POLICY_HH

#include <nv/common.hh>

namespace nv
{
	template < typename T >
	struct hash_table_no_extra_types_policy : false_type{};

	template < typename Key, typename Mapped, typename Hash, bool StoreHash >
	struct hash_table_standard_entry;

	template < typename Key, typename Mapped, typename Hash >
	struct hash_table_standard_entry< Key, Mapped, Hash, false >
	{
		typedef hash_table_standard_entry< Key, Mapped, Hash, false > this_type;
		typedef Key         key_type;
		typedef Mapped      mapped_type;
		typedef Hash        hash_type;
		typedef pair< const key_type, mapped_type > value_type;

		value_type value;

		static constexpr const key_type& get_key( const value_type& value ) { return value.first; }
		static constexpr const key_type& get_key( const this_type* entry ) { return entry->value.first; }
		static constexpr value_type* get_value( this_type* entry ) { return &(entry->value); }
	};

	template < typename Key, typename Mapped, typename Hash >
	struct hash_table_standard_entry< Key, Mapped, Hash, true >
	{
		typedef hash_table_standard_entry< Key, Mapped, Hash, true > this_type;
		typedef Key         key_type;
		typedef Mapped      mapped_type;
		typedef Hash        hash_type;
		typedef pair< const key_type, mapped_type > value_type;

		value_type value;
		hash_type  hash_code;

		static constexpr const key_type& get_key( const value_type& value ) { return value.first; }
		static constexpr const key_type& get_key( const this_type* entry ) { return entry->value.first; }
		static constexpr value_type* get_value( this_type* entry ) { return &( entry->value ); }
		static constexpr hash_type get_hash( const this_type* entry )
		{
			return entry->hash_code;
		}
		static inline void set_hash( this_type* entry, hash_type hash_code )
		{
			entry->hash_code = hash_code;
		}
	};

	template < typename EntryType, bool StoreHash >
	struct hash_table_standard_entry_hasher;

	template < typename EntryType >
	struct hash_table_standard_entry_hasher< EntryType, false >
	{
		typedef typename EntryType::hash_type hash_type;
		typedef typename EntryType::key_type  key_type;

		template < typename T >
		static constexpr hash_type get_hash( const T& value )
		{
			return hash< T, hash_type >::get( value );
		}
		static constexpr hash_type get_hash( const EntryType* entry )
		{
			return hash< key_type, hash_type >::get( EntryType::get_key( entry ) );
		}
		static inline void set_hash( EntryType*, hash_type ) {}
	};

	template < typename EntryType >
	struct hash_table_standard_entry_hasher< EntryType, true >
	{
		typedef typename EntryType::hash_type hash_type;

		template < typename T >
		static constexpr hash_type get_hash( const T& value )
		{
			return hash< T, hash_type >::get( value );
		}
		static constexpr hash_type get_hash( const EntryType* entry )
		{
			return EntryType::get_hash( entry );
		}
		static inline void set_hash( EntryType* entry, hash_type hash_code )
		{
			EntryType::set_hash( entry, hash_code );
		}
	};

	template < typename EntryType, typename HasherType, bool UseKey, bool UseHash >
	struct hash_table_standard_compare;

	template < typename EntryType, typename HasherType >
	struct hash_table_standard_compare< EntryType, HasherType, true, true >
	{
		template < typename T >
		static constexpr bool apply( const EntryType* entry, typename EntryType::hash_type h, const T& key )
		{
			return HasherType::get_hash( entry ) == h && EntryType::get_key( entry ) == key;
		}
	};

	template < typename EntryType, typename HasherType >
	struct hash_table_standard_compare< EntryType, HasherType, true, false >
	{
		template < typename T >
		static constexpr bool apply( const EntryType* entry, typename EntryType::hash_type, const T& key )
		{
			return EntryType::get_key( entry ) == key;
		}
	};

	template < typename EntryType, typename HasherType >
	struct hash_table_standard_compare< EntryType, HasherType, false, true >
	{
		template < typename T >
		static constexpr bool apply( const EntryType* entry, typename EntryType::hash_type h, const T& )
		{
			return HasherType::get_hash( entry ) == h;
		}
	};


	uint32 get_prime_larger_or_equal_to( uint32 value );

	struct hash_table_prime_rehash_policy
	{
		template< typename H >
		static enable_if_t< is_integral< H >::value, size_t >
		get_index( H h, size_t n ) { return h % n; }

		template< typename H >
		static enable_if_t< is_class< H >::value, size_t >
		get_index( H h, size_t n ) { return h.value() % n; }


		static uint32 get_bucket_count( uint32 requested_count )
		{
			return get_prime_larger_or_equal_to( requested_count );
		}
		static uint32 get_bucket_count( uint32 element_count, float load_factor )
		{
			uint32 min_count = static_cast<uint32>( element_count / load_factor );
			return get_prime_larger_or_equal_to( min_count );
		}
		static uint32 is_rehash_required( uint32 bucket_count, uint32 element_count, float load_factor )
		{
			uint32 min_count = static_cast<uint32>( element_count / load_factor );
			if ( bucket_count < 2 || min_count > bucket_count )
			{
				return get_prime_larger_or_equal_to( min_count );
			}
			return 0;
		}
	};

}

#endif // NV_STL_HASH_TABLE_POLICY_HH
