// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_LUA_VALUES_HH
#define NV_LUA_VALUES_HH

#include <nv/common.hh>
#include <nv/stl/type_traits/properties.hh>
#include <nv/stl/string.hh>

struct lua_State;

namespace nv
{
	namespace lua
	{
		class ref
		{
		public:
			static const int none  = -2;
			static const int nil   = -1;

			ref() : m_value( nil ) {}
			explicit ref( int lua_ref ) : m_value( lua_ref ) {}
			bool is_valid() const { return m_value >= 0; }
			int get() const { return m_value; }
		protected:
			int m_value;
		};

		typedef ptrdiff_t     linteger;
		typedef unsigned long lunsigned;
		typedef double        lnumber;

//  		struct passer
// 		{
// 			virtual void push( lua_State *L ) const = 0;
// 			virtual ~passer(){}
// 		};
// 
// 		template < typename T >
// 		struct returner
// 		{
// 			returner( lua_State *, int ) : value() {}
// 			returner( lua_State *, int, const T& ) : value() {}
// 			operator T() { return value; }
// 			operator const T() const { return value; }
// 			virtual returner<T> to( lua_State *L, int index ) = 0;
// 			virtual returner<T> to( lua_State *L, int index, const T& def ) = 0;
// 			virtual ~returner(){}
// 		protected:
// 			T value;
// 		};

		template < typename T >
		struct pass_traits
		{
			static void push( lua_State *L, const T& p ) { p.push( L ); }
			static T to( lua_State *L, int index ) { return T( L, index ); }
			static T to( lua_State *L, int index, const T& def ) { return T( L, index, def ); }
		};

		namespace detail
		{
			int upvalue_index( int i );

			void push_nil        ( lua_State *L );
			void push_unsigned   ( lua_State *L, lunsigned v );
			void push_integer    ( lua_State *L, linteger v );
			void push_number     ( lua_State *L, lnumber v );
			void push_bool       ( lua_State *L, bool v );
			void push_string_view( lua_State *L, string_view s );
			void push_cstring    ( lua_State *L, const char* s );
			void push_pointer    ( lua_State *L, void* p );
			void push_ref_object ( lua_State *L, ref object );

			lunsigned   to_unsigned   ( lua_State *L, int index );
			linteger    to_integer    ( lua_State *L, int index );
			lnumber     to_number     ( lua_State *L, int index );
			bool        to_bool       ( lua_State *L, int index );
//			const char* to_cstring    ( lua_State *L, int index );
			string_view to_string_view( lua_State *L, int index );
			void*       to_pointer    ( lua_State *L, int index );
			void*       to_ref_object ( lua_State *L, int index );

			lunsigned   to_unsigned   ( lua_State *L, int index, lunsigned def );
			linteger    to_integer    ( lua_State *L, int index, linteger def );
			lnumber     to_number     ( lua_State *L, int index, lnumber def );
			bool        to_bool       ( lua_State *L, int index, bool def );
//			const char* to_cstring    ( lua_State *L, int index, const char* def );
			string_view to_string_view( lua_State *L, int index, string_view def );
			void*       to_pointer    ( lua_State *L, int index, void* def );
			void*       to_ref_object ( lua_State *L, int index, void* def );

			void pop_and_discard( lua_State *L, int count );
			bool is_userdata( lua_State *L, int index, const char* metatable );
			void* check_userdata( lua_State *L, int index, const char* metatable );
			void* allocate_userdata( lua_State *L, size_t size, const char* metatable );
			template <typename T>
			void push_userdata( lua_State *L, const T& v, const char* metatable )
			{
				new (allocate_userdata(L, sizeof(T), metatable)) T(v);
			}
		}

		template <>
		struct pass_traits<linteger>
		{
			static void push( lua_State *L, linteger s ) { detail::push_integer( L, s ); }
			static linteger to( lua_State *L, int index ) { return detail::to_integer( L, index ); }
			static linteger to( lua_State *L, int index, linteger def ) { return detail::to_integer( L, index, def ); }
		};

		template <>
		struct pass_traits<lunsigned>
		{
			static void push( lua_State *L, lunsigned s ) { detail::push_unsigned( L, s ); }
			static lunsigned to( lua_State *L, int index ) { return detail::to_unsigned( L, index ); }
			static lunsigned to( lua_State *L, int index, lunsigned def ) { return detail::to_unsigned( L, index, def ); }
		};

		template <>
		struct pass_traits<lnumber>
		{
			static void push( lua_State *L, lnumber s ) { detail::push_number( L, s ); }
			static lnumber to( lua_State *L, int index ) { return detail::to_number( L, index ); }
			static lnumber to( lua_State *L, int index, lnumber def ) { return detail::to_number( L, index, def ); }
		};

		template <>
		struct pass_traits<bool>
		{
			static void push( lua_State *L, bool s ) { detail::push_bool( L, s ); }
			static bool to( lua_State *L, int index ) { return detail::to_bool( L, index ); }
			static bool to( lua_State *L, int index, bool def ) { return detail::to_bool( L, index, def ); }
		};

// 		template <>
// 		struct pass_traits<const char*>
// 		{
// 			static void push( lua_State *L, const char* s ) { detail::push_cstring( L, s ); }
// 			static const char* to( lua_State *L, int index ) { return detail::to_cstring( L, index ); }
// 			static const char* to( lua_State *L, int index, const char* def ) { return detail::to_cstring( L, index, def ); }
// 		};

		template <>
		struct pass_traits < string_view >
		{
			static void push( lua_State *L, string_view s ) { detail::push_string_view( L, s ); }
			static string_view to( lua_State *L, int index ) { return detail::to_string_view( L, index ); }
			static string_view to( lua_State *L, int index, string_view def ) { return detail::to_string_view( L, index, def ); }
		};

		template <>
		struct pass_traits < const_string >
		{
			static void push( lua_State *L, const const_string& s ) { detail::push_string_view( L, s ); }
			static const_string to( lua_State *L, int index ) { return const_string( detail::to_string_view( L, index ) ); }
			static const_string to( lua_State *L, int index, const string_view& def ) { return const_string( detail::to_string_view( L, index, def ) ); }
		};

		template <>
		struct pass_traits < string128 >
		{
			static void push( lua_State *L, const string128& s ) { detail::push_string_view( L, s ); }
			static string128 to( lua_State *L, int index ) { return string128( detail::to_string_view( L, index ) ); }
			static string128 to( lua_State *L, int index, const string_view& def ) { return string128( detail::to_string_view( L, index, def ) ); }
		};

		template <>
		struct pass_traits<ref>
		{
			static void push( lua_State *L, ref s ) { detail::push_ref_object( L, s ); }
		};


		namespace detail
		{

			template <typename T, class ENABLE = void >
			struct lua_type_impl
			{
				typedef T type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_floating_point< T >::value > >
			{
				typedef lnumber type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_integral< T >::value && is_signed< T >::value > >
			{
				typedef linteger type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_integral< T >::value && is_unsigned< T >::value > >
			{
				typedef lunsigned type;
			};

			template <typename T>
			struct lua_type_impl< T, enable_if_t< is_cstring< T >::value > >
			{
				typedef string_view type;
			};

			template <>
			struct lua_type_impl< bool >
			{
				typedef bool type;
			};

		}

		template < typename T >
		struct converted_type
		{
			typedef typename detail::lua_type_impl< remove_cvr_t<T> >::type type;
		};

		template < typename T >
		using converted_type_t = typename converted_type< T >::type;

		namespace detail
		{

 			template < typename T >
 			void push_value( lua_State *L, T&& p )
 			{
				pass_traits< converted_type_t<T> >::push( L, ::nv::forward<T>( p ) );
 			}

			template < typename T >
			void push_values(lua_State *L, T&& p)
			{
				push_value( L, forward<T>(p) );
			}
			template < typename T, typename ...Ts >
			void push_values(lua_State *L, T&& p, Ts&& ...ps)
			{
				push_value(L, forward<T>(p));
				push_values(L, forward<Ts>(ps)...);
			}


			template < typename T >
			inline void pop_value( lua_State *L, T& p )
			{
				p = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1 ) );
				detail::pop_and_discard(L, 1);
			}
			template < typename T >
			inline T pop_return_value( lua_State *L )
			{
				T ret = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1 ) );
				detail::pop_and_discard(L, 1);
				return ret;
			}

			template <>
			inline void pop_return_value<void>( lua_State* )
			{
			}

			template < typename T >
			inline void pop_value( lua_State *L, T& p, const T& def )
			{
				p = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1, def ) );
				detail::pop_and_discard(L, 1);
			}
			template < typename T >
			inline converted_type_t<T> pop_return_value( lua_State *L, const T& def )
			{
				T ret = static_cast<T>( pass_traits< converted_type_t<T> >::to( L, -1, def ) );
				detail::pop_and_discard(L, 1);
				return ret;
			}

			template < typename T >
			inline converted_type_t<T> get_value( lua_State *L, int index )
			{
				return pass_traits< converted_type_t<T> >::to( L, index );
			}

			template < typename T >
			inline converted_type_t<T> get_value( lua_State *L, int index, const T& def )
			{
				return pass_traits< converted_type_t<T> >::to( L, index, def );
			}

			template < typename T >
			struct return_count
			{
				const static int value = 1;
			};

			template <>
			struct return_count<void>
			{
				const static int value = 0;
			};

		}

	} // namespace lua

} // namespace nv

#endif // NV_LUA_VALUES_HH
