#include <nv/gfx/keyframed_mesh.hh>
#include <nv/gl/gl_device.hh>
#include <nv/sdl/sdl_window_manager.hh>
#include <nv/gfx/image.hh>
#include <nv/gfx/debug_draw.hh>
#include <nv/interface/context.hh>
#include <nv/interface/window.hh>
#include <nv/core/profiler.hh>
#include <nv/core/logging.hh>
#include <nv/core/logger.hh>
#include <nv/core/math.hh>
#include <nv/core/time.hh>
#include <nv/core/string.hh>
#include <glm/gtx/rotate_vector.hpp>

class application
{
public:
	application();
	bool initialize();
	bool run();
	~application();
protected:
	nv::mesh_data* generate_box();
	nv::mesh_data* generate_subdiv_sphere( int levels );
	nv::window_manager* m_wm;
	nv::device*      m_device;
	nv::window*      m_window;
	nv::context*     m_context;
	nv::debug_data*  m_debug_data;
	nv::texture      m_diffuse;
	nv::program      m_program;
	nv::mesh_data*   m_data;
	nv::vertex_array m_va;

	nv::clear_state   m_clear_state;
	nv::render_state  m_render_state;
	nv::scene_state   m_scene_state;
};

struct vertex
{
	nv::vec3 position;
};

nv::mesh_data* generate_subdiv_sphere( int levels )
{


	nv::mesh_raw_channel* vchannel = nv::mesh_raw_channel::create<vertex>( 8 );
	nv::mesh_raw_channel* ichannel = nv::mesh_raw_channel::create_index( nv::USHORT, 6 * 6 );
	vertex*     vtx = (vertex*)vchannel->data;
	nv::uint16* idx = (nv::uint16*)ichannel->data;

	nv::mesh_data* result = new nv::mesh_data;
	result->add_channel( vchannel );
	result->add_channel( ichannel );
	return result;

}

nv::mesh_data* application::generate_box()
{
	nv::mesh_raw_channel* vchannel = nv::mesh_raw_channel::create<vertex>( 8 );
	nv::mesh_raw_channel* ichannel = nv::mesh_raw_channel::create_index( nv::USHORT, 6 * 6 );
	vertex*     vtx = (vertex*)vchannel->data;
	nv::uint16* idx = (nv::uint16*)ichannel->data;
	
	vtx[0].position = nv::vec3( 1.0f, 1.0f, 1.0f );
	vtx[1].position = nv::vec3( 1.0f, 1.0f,-1.0f );
	vtx[2].position = nv::vec3(-1.0f, 1.0f,-1.0f );
	vtx[3].position = nv::vec3(-1.0f, 1.0f, 1.0f );
	vtx[4].position = nv::vec3( 1.0f,-1.0f, 1.0f );
	vtx[5].position = nv::vec3( 1.0f,-1.0f,-1.0f );
	vtx[6].position = nv::vec3(-1.0f,-1.0f,-1.0f );
	vtx[7].position = nv::vec3(-1.0f,-1.0f, 1.0f );

	auto square = [&]( nv::uint16 i, nv::uint16 a, nv::uint16 b, nv::uint16 c, nv::uint16 d ) 
	{
		idx[i+0] = a; 
		idx[i+1] = b; 
		idx[i+2] = c; 
		
		idx[i+3] = c; 
		idx[i+4] = d; 
		idx[i+5] = a; 
	};

	square( 0, 0, 1, 2, 3 );
	square( 6, 1, 0, 4, 5 );
	square(12, 2, 1, 5, 6 );
	square(18, 3, 2, 6, 7 );
	square(24, 0, 3, 7, 4 );
	square(30, 7, 6, 5, 4 );

	nv::mesh_data* result = new nv::mesh_data;
	result->add_channel( vchannel );
	result->add_channel( ichannel );
	return result;
}


application::application()
{
	NV_PROFILE( "app_construct" );
	m_wm      = new nv::sdl::window_manager;
	m_device  = new nv::gl_device();
	m_window  = m_wm->create_window( m_device, 1024, 800, false );
	m_context = m_window->get_context();

// 	nv::sampler sampler( nv::sampler::LINEAR, nv::sampler::REPEAT );
// 	nv::image_data* data = m_device->create_image_data( "data/manc.png" );
// 	m_diffuse  = m_device->create_texture( data, sampler );
// 	delete data;

	m_clear_state.buffers = nv::clear_state::COLOR_AND_DEPTH_BUFFER;
	m_clear_state.color   = nv::vec4(0.2f,0.2f,0.2f,1.0f);
	m_render_state.depth_test.enabled = true;
	m_render_state.culling.enabled    = true;
	m_render_state.blending.enabled   = false;
	m_render_state.blending.src_rgb_factor   = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_rgb_factor   = nv::blending::ONE_MINUS_SRC_ALPHA;
	m_render_state.blending.src_alpha_factor = nv::blending::SRC_ALPHA;
	m_render_state.blending.dst_alpha_factor = nv::blending::ONE_MINUS_SRC_ALPHA;

	m_debug_data = new nv::debug_data( m_context );
	m_debug_data->push_aabox( nv::vec3( 1.0f, 1.0f, 1.0f ), nv::vec3(-1.0f, -1.0f,-1.0f ), nv::vec3( 1.0f, 0.0f, 0.0f ) );
	m_debug_data->update();
}

bool application::initialize()
{
	NV_PROFILE( "app_initialize" );
	m_program = m_device->create_program( nv::slurp( "planet.vert" ), nv::slurp( "planet.frag" ) );
	m_data = generate_box();
	m_va   = m_context->create_vertex_array( m_data, nv::STATIC_DRAW );
	return true;
}

bool application::run()
{
	nv::profiler::pointer()->log_report(); 
	NV_PROFILE( "app_run" );
	int keypress = 0;

	nv::uint32 ticks   = m_wm->get_ticks();
	nv::uint32 last_ticks;
	nv::fps_counter_class< nv::system_us_timer > fps_counter;

	nv::uint16 count = 0;
	while(!keypress) 
	{
		last_ticks = ticks;
		ticks      = m_wm->get_ticks();
		nv::uint32 elapsed = ticks - last_ticks;
		m_context->clear( m_clear_state );
		glm::vec3 eye = glm::rotate( glm::vec3( 3.0f, 0.0f, 0.0f ), (ticks / 20.f), glm::vec3( 0.0,1.0,0.0 ) );
//		eye = glm::vec3( 3.0f, 0.0f, 0.0f );
		m_scene_state.set_model( nv::mat4(1.0f) );
		m_scene_state.get_camera().set_lookat(eye, glm::vec3(0.0f, 0.0f, 0.0f), glm::vec3(0.0, 1.0, 0.0));
		float ratio = (float)m_window->get_width() / (float)m_window->get_height();
		m_scene_state.get_camera().set_perspective(60.0f, ratio, 0.1f, 1000.0f);

//		m_context->bind( m_diffuse, nv::TEX_DIFFUSE );
		m_device->set_opt_uniform( m_program, "center", glm::vec3(0.0,0.0,0.0) );
		m_device->set_opt_uniform( m_program, "radius", 1.0f );
		m_device->set_opt_uniform( m_program, "light_position", glm::vec3(120.0, 120.0, 0) );
		m_device->set_opt_uniform( m_program, "light_diffuse",  glm::vec4(0.5,0.5,0.5,1.0) );
		m_device->set_opt_uniform( m_program, "light_specular", glm::vec4(1.0,1.0,1.0,1.0) );
		m_render_state.culling.order    = nv::culling::CW;
		m_context->draw( nv::TRIANGLES, m_render_state, m_scene_state, m_program, m_va, 6*6 );
		m_render_state.culling.order    = nv::culling::CCW;

		m_context->draw( nv::LINES, m_render_state, m_scene_state, m_debug_data->get_program(), m_debug_data->get_vertex_array(), m_debug_data->get_count() );

		m_window->swap_buffers();

		nv::io_event event;
		while(m_window->poll_event(event)) 
		{      
			switch (event.type) 
			{
			case nv::EV_QUIT:
				keypress = 1;
				break;
			case nv::EV_KEY:
				if (event.key.pressed)
				{
					switch (event.key.code) 
					{
					case nv::KEY_ESCAPE : keypress = 1; break;
					case nv::KEY_F1 : nv::profiler::pointer()->log_report(); break;
					default: break;
					}
				}
				break;
			default: break;
			}
		}

		fps_counter.tick();
		if ( count % 120 == 0 )
		{
			m_window->set_title( "Nova Engine - " + nv::to_string( (nv::uint32)fps_counter.fps() ) );
		}
		count++;
	}
	return true;
}

application::~application()
{
	m_context->release( m_va );
	m_device->release( m_program );
	m_device->release( m_diffuse );

	delete m_window;
	delete m_device;
}


int main(int, char* [])
{
	nv::logger log(nv::LOG_TRACE);
	log.add_sink( new nv::log_file_sink("log.txt"), nv::LOG_TRACE );
	log.add_sink( new nv::log_console_sink(), nv::LOG_TRACE );
	
	NV_LOG( nv::LOG_NOTICE, "Logging started" );
	application app;
	if ( app.initialize() )
	{
		app.run();
	}
	NV_LOG( nv::LOG_NOTICE, "Logging stopped" );

	return 0;
}

