// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_STL_STRING_STRING_BASE_HH
#define NV_STL_STRING_STRING_BASE_HH

#include <nv/stl/string/common.hh>
#include <nv/stl/functional/hash.hh>
#include <nv/stl/memory.hh> // TODO: /stl/memory/array_ref?

namespace nv
{

	size_t sint32_to_buffer( array_ref< char > buffer, sint32 n );
	size_t sint64_to_buffer( array_ref< char > buffer, sint64 n );
	size_t uint32_to_buffer( array_ref< char > buffer, uint32 n );
	size_t uint64_to_buffer( array_ref< char > buffer, uint64 n );
	size_t f32_to_buffer( array_ref< char > buffer, f32 n );
	size_t f64_to_buffer( array_ref< char > buffer, f64 n );

	sint32 buffer_to_sint32( const char* str, char** end );
	sint64 buffer_to_sint64( const char* s, char** end );
	uint32 buffer_to_uint32( const char* s, char** end );
	uint64 buffer_to_uint64( const char* s, char** end );
	float buffer_to_f32( const char* s, char** end );
	double buffer_to_f64( const char* s, char** end );
	
	// string base class - will become a base for a string class later
	template < typename Storage >
	class string_base : public Storage
	{
		typedef Storage                                     base_type;
		typedef string_base< base_type >                    this_type;
	public:
		typedef Storage                                     storage_type;
		typedef typename base_type::value_type              value_type;
		typedef typename base_type::pointer                 pointer;
		typedef typename base_type::const_pointer           const_pointer;
		typedef typename base_type::reference               reference;
		typedef typename base_type::const_reference         const_reference;
		typedef typename base_type::iterator                iterator;
		typedef typename base_type::const_iterator          const_iterator;
		typedef typename base_type::reverse_iterator        reverse_iterator;
		typedef typename base_type::const_reverse_iterator  const_reverse_iterator;
		typedef typename base_type::size_type               size_type;
		typedef typename base_type::difference_type         difference_type;

		static constexpr size_type npos = size_type( -1 );

		inline size_type length()   const { return this->size(); }

		// string operations
		int compare( const string_view& rhs ) const;
		bool starts_with( value_type c ) const;
		bool starts_with( const string_view& s ) const;
		bool ends_with( value_type c ) const;
		bool ends_with( const string_view& s ) const;
		auto find( value_type c, size_type pos = 0 ) const->size_type;
		auto find( const string_view& s, size_type pos = 0 ) const->size_type;
		auto rfind( value_type c, size_type pos = 0 ) const->size_type;
		auto rfind( const string_view& s, size_type pos = 0 ) const->size_type;
		auto find_first_of( value_type c ) const->size_type;
		auto find_first_of( const string_view& s ) const->size_type;
		auto find_last_of( value_type c )  const->size_type;
		auto find_last_of( const string_view& s ) const->size_type;
		auto find_first_not_of( value_type c ) const->size_type;
		auto find_first_not_of( const string_view& s ) const->size_type;
		auto find_last_not_of( value_type c ) const->size_type;
		auto find_last_not_of( const string_view& s ) const->size_type;

		// string operations
		string_view substr( size_type p, size_type n = npos ) const;
		string_view without_prefix( size_type p ) const;
		string_view without_suffix( size_type p ) const;

		template < typename H = size_type >
		inline H get_hash() const
		{
			return hash_string< H >( this->data(), this->size() );
		}

	protected:
		using base_type::base_type;

		template < typename ReverseIterator >
		size_type reverse_distance( ReverseIterator first, ReverseIterator last ) const
		{
			return this->size() - 1 - static_cast<size_t>( nv::distance( first, last ) );
		}
	};

	template< typename T, typename H >
	struct hash< T, H, typename enable_if< is_string_base<T>::value, void >::type >
		: detail::hash_base< T, H >
	{
		static H get( const T& value )
		{
			return value.template get_hash< H >();
		}
		inline H operator()( const T& value ) const { return get( value ); }
	};

	class string_view : public string_base< array_view< char > >
	{
	public:
		typedef string_base< array_view< char > > this_type;

		constexpr string_view() : this_type() {}
		inline string_view( const string_view& rhs ) : this_type( rhs.data(), rhs.size() ) {}
		template < typename S >
		inline string_view( const string_base<S>& rhs ) : this_type( rhs.data(), rhs.size() ) {}
		constexpr string_view( const char* str, size_type len ) : this_type( str, len ) {}

		// Literal constructors
		template< size_t N >
		constexpr string_view( char( &s )[N] ) : this_type( s, N - 1 ) {}
		template< size_t N >
		constexpr string_view( const char( &s )[N] ) : this_type( s, N - 1 ) {}

		// Non-literal constructors
		template< typename U >
		inline string_view( U str, typename enable_if<is_same<U, const char*>::value>::type* = nullptr ) : this_type( str, nvstrlen( str ) ) {}

		// Non-literal constructors
		template< typename U >
		inline string_view( U str, typename enable_if<is_same<U, char*>::value>::type* = nullptr ) : this_type( str, nvstrlen( str ) ) {}

		inline string_view& operator=( const string_view &rhs )
		{
			assign( rhs.data(), rhs.size() );
			return *this;
		}

		// modifiers
		inline void clear()
		{
			assign( nullptr, 0 );
		}
		inline void remove_prefix( size_type n )
		{
			size_type s = size();
			if ( n > s ) n = s;
			assign( data() + n, s - n );
		}
		inline void remove_suffix( size_type n )
		{
			size_type s = size();
			if ( n > s ) n = s;
			assign( data(), s - n );
		}

	};

	template < typename Storage >
	inline int string_base< Storage >::compare( const string_view& rhs ) const
	{
		size_type this_size = this->size();
		int cmp = nvmemcmp( this->data(), rhs.data(), nv::min( this_size, rhs.size() ) );
		return cmp != 0 ? cmp : ( this_size == rhs.size() ? 0 : this_size < rhs.size() ? -1 : 1 );
	}

	template < typename Storage >
	inline bool string_base< Storage >::starts_with( value_type c ) const
	{
		return !this->empty() && c == this->front();
	}
	template < typename Storage >
	inline bool string_base< Storage >::starts_with( const string_view& s ) const
	{
		return this->size() >= s.size() && nvmemcmp( this->data(), s.data(), s.size() ) == 0;
	}

	template < typename Storage >
	inline bool string_base< Storage >::ends_with( value_type c ) const
	{
		return !this->empty() && c == this->back();
	}
	template < typename Storage >
	inline bool string_base< Storage >::ends_with( const string_view& s ) const
	{
		return this->size() >= s.size() && nvmemcmp( this->data() + this->size() - s.size(), s.data(), s.size() ) == 0;
	}

	template < typename Storage >
	inline auto string_base< Storage >::find( value_type c, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_iterator it = nv::find_if( this->cbegin() + static_cast<difference_type>( pos ), this->cend(), [=] ( value_type val ) { return val == c; } );
		return it == this->cend() ? npos : static_cast<size_type>( nv::distance( this->cbegin(), it ) );
	}
	template < typename Storage >
	inline auto string_base< Storage >::find( const string_view& s, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_iterator it = nv::search( this->cbegin() + static_cast<difference_type>( pos ), this->cend(), s.cbegin(), s.cend() );
		return it == this->cend() ? npos : static_cast<size_type>( nv::distance( this->cbegin(), it ) );
	}

	template < typename Storage >
	inline auto string_base< Storage >::rfind( value_type c, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_reverse_iterator it = nv::find_if( this->crbegin() + static_cast<difference_type>( pos ), this->crend(), [=] ( value_type val ) { return val == c; } );
		return it == this->crend() ? npos : this->reverse_distance( this->crbegin(), it );
	}
	template < typename Storage >
	inline auto string_base< Storage >::rfind( const string_view& s, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_reverse_iterator it = nv::search( this->crbegin() + static_cast<difference_type>( pos ), this->crend(), s.crbegin(), s.crend() );
		return it == this->crend() ? npos : this->reverse_distance( this->crbegin(), it );
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_first_of( value_type c ) const -> size_type
	{
		return this->find( c );
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_first_of( const string_view& s ) const -> size_type
	{
		const_iterator it = nv::find_first_of( this->cbegin(), this->cend(), s.cbegin(), s.cend() );
		return it == this->cend() ? npos : static_cast<size_type>( nv::distance( this->cbegin(), it ) );
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_last_of( value_type c ) const -> size_type
	{
		return this->rfind( c );
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_last_of( const string_view& s ) const -> size_type
	{
		const_reverse_iterator it = nv::find_first_of( this->crbegin(), this->crend(), s.cbegin(), s.cend() );
		return it == this->crend() ? npos : this->reverse_distance( this->crbegin(), it );
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_first_not_of( value_type c ) const -> size_type
	{
		for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
			if ( c != *it )
			return static_cast<size_type>( nv::distance( this->cbegin(), it ) );
		return npos;
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_first_not_of( const string_view& s ) const -> size_type
	{
		for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
			if ( 0 == nvmemchr( s.data(), static_cast<uchar8>( *it ), s.size() ) )
			return static_cast<size_type>( nv::distance( this->cbegin(), it ) );
		return npos;
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_last_not_of( value_type c ) const -> size_type
	{
		for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
			if ( c != *it )
			return this->reverse_distance( this->crbegin(), it );
		return npos;
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_last_not_of( const string_view& s ) const -> size_type
	{
		for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
			if ( 0 == nvmemchr( s.data(), static_cast<uchar8>( *it ), s.size() ) )
			return this->reverse_distance( this->crbegin(), it );
		return npos;
	}

	template < typename Storage >
	inline string_view string_base< Storage >::substr( size_type p, size_type n ) const
	{
		if ( p > this->size() ) return string_view(); // NV_THROW( out_of_range( "substr" ) );
		if ( n == npos || n == p || p + n > this->size() ) n = this->size() - p;
		return string_view( this->data() + p, n );
	}

	template < typename Storage >
	inline string_view string_base< Storage >::without_prefix( size_type p ) const
	{
		string_view result( this->data(), this->size() );
		result.remove_prefix( p );
		return result;
	}

	template < typename Storage >
	inline string_view string_base< Storage >::without_suffix( size_type p ) const
	{
		string_view result( this->data(), this->size() );
		result.remove_suffix( p );
		return result;
	}


#define NV_STRING_BASE_CAST_OPERATORS( OPERATOR )\
template < typename S > inline bool operator OPERATOR ( const string_base< S >& lhs, const char* rhs )\
{\
	return lhs OPERATOR string_view( rhs );\
}\
template < typename S > inline bool operator OPERATOR ( const char* lhs, const string_base< S >& rhs )\
{\
	return string_view( lhs ) OPERATOR rhs;\
}\

	// Base operators
	template < typename S1, typename S2 >
	inline bool operator==( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.size() != rhs.size() ? false : ( lhs.compare( rhs ) == 0 );
	}
	template < typename S1, typename S2 >
	inline bool operator!=( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.size() != rhs.size() ? true : ( lhs.compare( rhs ) != 0 );
	}
	template < typename S1, typename S2 >
	inline bool operator<( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) < 0;
	}
	template < typename S1, typename S2 >
	inline bool operator>( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) > 0;
	}
	template < typename S1, typename S2 >
	inline bool operator<=( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) <= 0;
	}
	template < typename S1, typename S2 >
	inline bool operator>=( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) >= 0;
	}

	NV_STRING_BASE_CAST_OPERATORS( == )
	NV_STRING_BASE_CAST_OPERATORS( != )
	NV_STRING_BASE_CAST_OPERATORS( < )
	NV_STRING_BASE_CAST_OPERATORS( > )
	NV_STRING_BASE_CAST_OPERATORS( <= )
	NV_STRING_BASE_CAST_OPERATORS( >= )

}

#undef NV_STRING_BASE_CAST_OPERATORS

#endif // NV_STL_STRING_STRING_BASE_HH
