// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file pair.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief STL pair library
 */

#ifndef NV_STL_UTILITY_PAIR_HH
#define NV_STL_UTILITY_PAIR_HH

#include <nv/stl/utility/common.hh>

namespace nv
{

	template < typename T1, typename T2 >
	struct pair
	{
		typedef pair< T1, T2 > this_type;
		typedef T1 first_type;
		typedef T2 second_type;

		constexpr pair() : first(), second() {}
		constexpr pair( const first_type& f, const second_type& s )
			: first( f ), second( s ) {}

		template < typename U1, typename U2 >
		constexpr pair( U1&& f, U2&& s )
			: first( ::nv::forward<U1>( f ) ), second( ::nv::forward<U2>( s ) ) {}
		template < typename U1, typename U2 >
		constexpr pair( const U1& f, U2&& s )
			: first( f ), second( ::nv::forward<U2>( s ) ) {}
		template < typename U1, typename U2 >
		constexpr pair( U1&& f, const U2& s )
			: first( ::nv::forward<U1>( f ) ), second( s ) {}


		pair( const pair& ) = default;
		pair( pair&& ) = default;

		this_type& operator= ( this_type&& rhs )
		{
			first = ::nv::forward<first_type>( rhs.first );
			second = ::nv::forward<first_type>( rhs.second );
		}

		this_type& operator= ( const this_type& rhs )
		{
			first = rhs.first;
			second = rhs.second;
		}

		void swap( this_type& rhs )
		{
			if ( this != &rhs )
			{
				swap( first, rhs.first );
				swap( second, rhs.second );
			}
		}

		first_type  first;
		second_type second;
	};

	template < typename T1, typename T2 >
	void swap( pair< T1, T2 >& lhs, pair< T1, T2 >& rhs )
	{
		lhs.swap( rhs );
	}

	template < typename T1, typename T2 >
	constexpr bool operator== ( const pair< T1, T2 >& lhs, const pair< T1, T2 >& rhs )
	{
		return lhs.first == rhs.first && lhs.second == rhs.second;
	}

	template < typename T1, typename T2 >
	constexpr bool operator!= ( const pair< T1, T2 >& lhs, const pair< T1, T2 >& rhs )
	{
		return !( lhs == rhs );
	}

	template < typename T1, typename T2 >
	constexpr bool operator< ( const pair< T1, T2 >& lhs, const pair< T1, T2 >& rhs )
	{
		return lhs.first < rhs.first || ( !( rhs.first < lhs.first ) && lhs.second < rhs.second );
	}

	template < typename T1, typename T2 >
	constexpr bool operator> ( const pair< T1, T2 >& lhs, const pair< T1, T2 >& rhs )
	{
		return ( rhs < lhs );
	}

	template < typename T1, typename T2 >
	constexpr bool operator<= ( const pair< T1, T2 >& lhs, const pair< T1, T2 >& rhs )
	{
		return !( rhs < lhs );
	}

	template < typename T1, typename T2 >
	constexpr bool operator>= ( const pair< T1, T2 >& lhs, const pair< T1, T2 >& rhs )
	{
		return !( lhs < rhs );
	}

}

#endif // NV_STL_UTILITY_PAIR_HH
