// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file logical_ops.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief functional logical function objects
*/

#ifndef NV_STL_FUNCTIONAL_LOGICAL_OPS_HH
#define NV_STL_FUNCTIONAL_LOGICAL_OPS_HH

#include <nv/stl/functional/common.hh>

namespace nv
{

	template< typename T = void >
	struct logical_and
	{
		typedef T result_type;
		typedef T first_argument_type;
		typedef T second_argument_type;

		constexpr bool operator() ( const T& lhs, const T& rhs ) const
		{
			return lhs && rhs;
		}
	};

	template<>
	struct logical_and< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T1, typename T2 >
		constexpr auto operator() ( T1&& lhs, T2&& rhs ) const
			-> decltype( static_cast<T1&&>( lhs ) && static_cast<T2&&>( rhs ) )
		{
			return static_cast<T1&&>( lhs ) && static_cast<T2&&>( rhs );
		}
	};

	template< typename T = void >
	struct logical_or
	{
		typedef T result_type;
		typedef T first_argument_type;
		typedef T second_argument_type;

		constexpr bool operator() ( const T& lhs, const T& rhs ) const
		{
			return lhs || rhs;
		}
	};

	template<>
	struct logical_or< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T1, typename T2 >
		constexpr auto operator() ( T1&& lhs, T2&& rhs ) const
			-> decltype( static_cast<T1&&>( lhs ) || static_cast<T2&&>( rhs ) )
		{
			return static_cast<T1&&>( lhs ) || static_cast<T2&&>( rhs );
		}
	};

	template< typename T = void >
	struct logical_not
	{
		typedef T result_type;
		typedef T argument_type;

		constexpr T operator() ( const T& arg ) const
		{
			return !arg;
		}
	};

	template<>
	struct logical_not< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T >
		constexpr auto operator() ( T&& arg ) const
			-> decltype( !static_cast<T&&>( arg ) )
		{
			return !static_cast<T&&>( arg );
		}
	};

}

#endif // NV_STL_FUNCTIONAL_LOGICAL_OPS_HH
