// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/common.hh"
#include "nv/stl/range.hh"
#include "nv/core/logging.hh"
#include "nv/lib/gl.hh"

#if defined( NV_GL_DYNAMIC )

#include "nv/core/library.hh"

#if defined( NV_SDL_GL )
#	include "nv/lib/sdl.hh"
#endif

// for wgl support
#if NV_PLATFORM == NV_WINDOWS
#include <windows.h>
#endif

#define NV_GL_FUN( rtype, fname, fparams ) rtype (NV_GL_APIENTRY *fname) fparams = nullptr;
#define NV_GL_FUN_REN( rtype, fname, rname, fparams ) rtype (NV_GL_APIENTRY *rname) fparams = nullptr;
#define NV_GL_FUN_EXT NV_GL_FUN
#include <nv/lib/detail/gl_functions.inc>
#if NV_PLATFORM == NV_WINDOWS
#include <nv/lib/detail/wgl_functions.inc>
#endif
#include <nv/lib/detail/gl_ext/gl_ext_all_functions.inc>
#undef NV_GL_FUN_REN
#undef NV_GL_FUN_EXT
#undef NV_GL_FUN

static nv::library gl_library;
static nv::gl_extensions gl_loaded_extensions = nv::gl_extensions(0);
static void* (NV_GL_APIENTRY *gl_ext_loader) ( const char* ) = nullptr;
static bool gl_library_loaded = false;
static bool wgl_library_loaded = false;

static const char *gl_extension_names[] = {
	"UNKNOWN",
#define NV_GL_EXTENSION( count, id, name ) "GL_EXT_"#id,
#include <nv/lib/detail/gl_ext/gl_ext_info.inc>
#undef NV_GL_EXTENSION
};

// static const char *gl_extension_ids[] = {
// 	"UNKNOWN",
// #define NV_GL_EXTENSION( count, id, name ) #id,
// #include <nv/lib/detail/gl_ext/gl_ext_info.inc>
// #undef NV_GL_EXTENSION
// };


static void* load_gl_ext_symbol_impl( const char* name, nv::log_level fail_level = nv::LOG_DEBUG )
{
	void* result = gl_ext_loader( name );
	NV_LOG( ( result ? nv::LOG_DEBUG : fail_level ), "load_gl_ext_symbol : ", name, ( result ? " succeded." : "failed." ) );
	return result;
}

static void* load_gl_ext_symbol( const char* name, bool iterate, const char* ext )
{
	void * result        = nullptr;
	result = load_gl_ext_symbol_impl( ext ? ( std::string(name) + ext ).c_str() : name );
	if ( result ) return result;
	if ( iterate )
	{
		result = gl_ext_loader( (std::string(name) + "ARB").c_str() );
		if ( result ) return result;
		result = gl_ext_loader( (std::string(name) + "EXT").c_str() );
		if ( result ) return result;
	}
	return result;
}

bool nv::load_gl_library( const char* path, bool force_reload )
{
	if ( gl_library_loaded && !force_reload ) return true;
#if defined( NV_SDL_GL )
#		define NV_GL_LOAD( symbol ) void_assign( symbol, SDL_GL_GetProcAddress(#symbol) );
#		define NV_GL_LOAD_EXT( symbol ) void_assign( symbol, SDL_GL_GetProcAddress(#symbol) );
	void_assign( gl_ext_loader, SDL_GL_GetProcAddress );
#else
	if ( !gl_library.is_open() ) gl_library.open( path );

#	if NV_PLATFORM == NV_WINDOWS 
#		define NV_GL_LOAD( symbol ) void_assign( symbol, gl_library.get(#symbol) );
		void_assign( gl_ext_loader, gl_library.get( "wglGetProcAddress" ) );
#		define NV_GL_LOAD_EXT( symbol ) void_assign( symbol, gl_ext_loader(#symbol) );
#	elif (NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE)
#		define NV_GL_LOAD( symbol ) void_assign( symbol, gl_library.get(#symbol) );
		void_assign( gl_ext_loader, gl_library.get("glXGetProcAddress") );
#		define NV_GL_LOAD_EXT( symbol ) void_assign( symbol, gl_ext_loader(#symbol) );
#	else
#		define NV_GL_LOAD( symbol ) void_assign( symbol, gl_library.get(#symbol) );
#		define NV_GL_LOAD_EXT( symbol ) void_assign( symbol, gl_library.get(#symbol) );
#	endif
#endif

#	define NV_GL_FUN( rtype, fname, fparams ) NV_GL_LOAD( fname )
#	define NV_GL_FUN_EXT( rtype, fname, fparams ) NV_GL_LOAD_EXT( fname )
#	include <nv/lib/detail/gl_functions.inc>
#	undef NV_GL_FUN_EXT
#	undef NV_GL_FUN

#	undef NV_GL_LOAD
#	undef NV_GL_LOAD_EXT
	gl_library_loaded = true;
	return true;
}



bool nv::load_wgl_library( const char* path /*= NV_GL_PATH */, bool force_reload )
{
	if ( wgl_library_loaded && !force_reload ) return true;
#if NV_PLATFORM == NV_WINDOWS 
#if defined( NV_SDL_GL )
#		define NV_GL_LOAD( symbol ) void_assign( symbol, SDL_GL_GetProcAddress(#symbol) );
#		define NV_GL_LOAD_EXT( symbol ) void_assign( symbol, SDL_GL_GetProcAddress(#symbol) );
#	    define NV_GL_LOAD_REN( fname, rname )  void_assign( rname, SDL_GL_GetProcAddress(#fname) );
	void_assign( gl_ext_loader, SDL_GL_GetProcAddress );
#else // 
	if ( !gl_library.is_open() ) gl_library.open( path );

	void_assign( gl_ext_loader, gl_library.get("wglGetProcAddress") );
#define NV_GL_LOAD( symbol ) void_assign( symbol, gl_library.get(#symbol) );
#define NV_GL_LOAD_EXT( symbol ) void_assign( symbol, gl_ext_loader(#symbol) );
#define NV_GL_LOAD_REN( fname, rname ) void_assign( rname, gl_library.get(#fname) );
#endif 
#	define NV_GL_FUN( rtype, fname, fparams ) NV_GL_LOAD( fname )
#	define NV_GL_FUN_EXT( rtype, fname, fparams ) NV_GL_LOAD_EXT( fname )
#	define NV_GL_FUN_REN( rtype, fname, rname, fparams ) NV_GL_LOAD_REN( fname, rname )
#	include <nv/lib/detail/wgl_functions.inc>
#	undef NV_GL_FUN_REN
#	undef NV_GL_FUN_EXT
#	undef NV_GL_FUN

#	undef NV_GL_LOAD
#	undef NV_GL_LOAD_EXT
#	undef NV_GL_LOAD_REN
	wgl_library_loaded = true;
	return true;
#else
	return false;
#endif
}

#endif // NV_DYNAMIC

bool nv::load_gl_no_context( const char* path /*= NV_GL_PATH */ )
{
#if NV_PLATFORM == NV_WINDOWS 
	if ( !gl_library.is_open() ) gl_library.open( path );
	if ( wgl_library_loaded ) return true;

	HGLRC (NV_GL_APIENTRY *wgl_createcontext) (HDC)        = nullptr;
	BOOL  (NV_GL_APIENTRY *wgl_makecurrent)   (HDC, HGLRC) = nullptr;
	BOOL  (NV_GL_APIENTRY *wgl_deletecontext) (HGLRC)      = nullptr;

	void_assign( wgl_createcontext, gl_library.get("wglCreateContext") );
	void_assign( wgl_makecurrent,   gl_library.get("wglMakeCurrent") );
	void_assign( wgl_deletecontext, gl_library.get("wglDeleteContext") );

	WNDCLASS wndClass;
	HINSTANCE hInstance = 0;

	ZeroMemory(&wndClass, sizeof(WNDCLASS));

	wndClass.cbClsExtra = 0;
	wndClass.cbWndExtra = 0;			
	wndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
	wndClass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	wndClass.hInstance = hInstance;
	wndClass.lpfnWndProc = reinterpret_cast<WNDPROC>( DefWindowProc );
	wndClass.lpszClassName = TEXT("Dummy67789");
	wndClass.lpszMenuName = 0;
	wndClass.style = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;

	/*DWORD err = */GetLastError();
	RegisterClass(&wndClass);
	/*err = */GetLastError();


	HWND hWndFake = CreateWindow(TEXT("Dummy67789"), "FAKE", WS_OVERLAPPEDWINDOW | WS_MAXIMIZE | WS_CLIPCHILDREN, 
		0, 0, CW_USEDEFAULT, CW_USEDEFAULT, NULL, 
		NULL, GetModuleHandle(nullptr), NULL); 

	HDC hDC = GetDC(hWndFake); 

	PIXELFORMATDESCRIPTOR pfd; 
	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR)); 
	pfd.nSize= sizeof(PIXELFORMATDESCRIPTOR); 
	pfd.nVersion   = 1; 
	pfd.dwFlags    = PFD_DOUBLEBUFFER | PFD_SUPPORT_OPENGL | PFD_DRAW_TO_WINDOW; 
	pfd.iPixelType = PFD_TYPE_RGBA; 
	pfd.cColorBits = 32; 
	pfd.cDepthBits = 24; 
	pfd.iLayerType = PFD_MAIN_PLANE; 

	int iPixelFormat = ChoosePixelFormat(hDC, &pfd); 
	if (iPixelFormat == 0)return false; 

	if(!SetPixelFormat(hDC, iPixelFormat, &pfd))return false; 

	HGLRC hRCFake = wgl_createcontext(hDC); 
	wgl_makecurrent(hDC, hRCFake); 

	LoadLibrary( "gdi32.dll" );
	bool gl_loaded = nv::load_gl_library( path );
	bool wgl_loaded = nv::load_wgl_library( path );
	bool result = gl_loaded && wgl_loaded;

	wgl_makecurrent(NULL, NULL); 
	wgl_deletecontext(hRCFake); 
	DestroyWindow(hWndFake); 
	return result;
#else
	return false;
#endif
}

const char* nv::get_gl_extension_name( gl_extensions extension )
{
	uint32 value = uint32( extension );
	uint32 index = 0; 
	while ( value >>= 1 ) index++;
	return NV_SAFE_ARRAY(gl_extension_names, index+1, gl_extension_names[0] );
}

bool nv::load_gl_extension( gl_extensions extension )
{
	const char* name = get_gl_extension_name( extension );
	// TODO: first check for support using gl mechanisms 
	//       see SDL 2.0 SDL_video.c
	if ( !gl_library.is_open() ) 
	{
		NV_LOG_ERROR( "load_gl_extension used, while gl_library was closed!" );
		return false;
	}

	if ( gl_ext_loader == nullptr )
	{
		NV_LOG_ERROR( "load_gl_extension used, while gl_ext_loader was undefined!" );
		return false;
	}
	NV_LOG_DEBUG( "load_gl_extension - loading extension - \"", name, "\"..." );

	uint32 count      = 0; 
	uint32 fail_count = 0;

#	define NV_GL_FUN_EXT( rtype, symbol, fparams ) \
	void_assign( symbol, load_gl_ext_symbol(#symbol, true, nullptr) ); \
	count++; if ( !symbol ) fail_count++;

	switch ( extension )
	{
	case GL_EXT_FRAMEBUFFER_BLIT : {
#include <nv/lib/detail/gl_ext/gl_ext_framebuffer_blit_functions.inc>
	} break;
	case GL_EXT_FRAMEBUFFER_OBJECT : {
#include <nv/lib/detail/gl_ext/gl_ext_framebuffer_object_functions.inc>
	} break;
	default : {
		NV_LOG_ERROR( "load_gl_extension - unknown extension \"", name, "\"!" );
		return false;
	}
	}
#	undef NV_GL_FUN_EXT

	if ( fail_count == 0 )
	{
		NV_LOG_NOTICE( "load_gl_extension - extension \"", name, "\" loaded (", count, " symbols)" );
		gl_loaded_extensions = gl_extensions( gl_loaded_extensions | static_cast<unsigned>( extension ) );
		return false;
	}
	NV_LOG_NOTICE( "load_gl_extension - failed to load extension \"", name, "\" (", count, "/", fail_count, " symbols loaded)" );
	return true;
}

nv::gl_extensions nv::load_gl_extensions( uint32 extensions )
{
	gl_extensions result = gl_extensions(0);
	for ( auto ext : nv::bits( gl_extensions(extensions) ) )
	{
		if ( load_gl_extension(ext) ) result = gl_extensions( result | ext );
	}
	return result;
}

bool nv::is_gl_extension_loaded( gl_extensions extensions )
{
	return ( gl_loaded_extensions & extensions ) != 0;
}

bool nv::are_gl_extensions_loaded( uint32 extensions )
{
	for ( auto ext : nv::bits( gl_extensions(extensions) ) )
	{
		if ( !is_gl_extension_loaded(ext) ) return false;
	}
	return true;
}

