// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/gfx/mesh_creator.hh"

#include "nv/interface/data_channel_access.hh"

struct nv_key_transform { nv::transform tform; };

void nv::mesh_nodes_creator::pre_transform_keys()
{
	if ( m_data->m_flat ) return;
	merge_keys();
	uint32 max_frames = 0;
	for ( size_t i = 0; i < m_data->get_count(); ++i )
	{
		sint16 parent_id = m_data->m_nodes[i].parent_id;
		data_channel_set* keys   = m_data->m_nodes[i].data;
		data_channel_set* pkeys  = ( parent_id != -1 ? m_data->m_nodes[parent_id].data : nullptr );
		size_t count     = ( keys ? keys->get_channel_size(0) : 0 );
		size_t pcount    = ( pkeys ? pkeys->get_channel_size(0) : 0 );
		max_frames = nv::max<uint32>( count, max_frames );
		if ( pkeys && pkeys->size() > 0 && keys && keys->size() > 0 )
		{
			data_channel_access< nv_key_transform > channel_creator( keys, 0 );
			nv_key_transform* channel = channel_creator.data();
			const nv_key_transform* pchannel = pkeys->get_channel(0)->data_cast< nv_key_transform >();
			for ( unsigned n = 0; n < count; ++n )
			{
				channel[n].tform = pchannel[ nv::min( n, pcount-1 ) ].tform * channel[n].tform;
			}
		}
	}

	// DAE pre_transform hack
	if ( m_data->m_frame_rate == 1 )
	{
		m_data->m_frame_rate = 32;
		m_data->m_duration   = static_cast<float>( max_frames );
	}

	m_data->m_flat = true;
}


void nv::mesh_nodes_creator::merge_keys()
{
	for ( size_t i = 0; i < m_data->get_count(); ++i )
	{
		data_channel_set* old_keys = m_data->m_nodes[i].data;
		if ( old_keys && old_keys->size() > 0 )
		{
			size_t chan_count = old_keys->size();
			if ( chan_count == 1 
				&& old_keys->get_channel(0)->descriptor().size() == 1
				&& old_keys->get_channel(0)->descriptor()[0].etype == TRANSFORM ) continue;

			size_t max_keys = 0;
			for ( size_t c = 0; c < chan_count; ++c )
			{
				max_keys = nv::max( max_keys, old_keys->get_channel(c)->size() );
			}

			data_channel_set* new_keys = data_channel_set_creator::create( 1 );
			data_channel_set_creator nk_access( new_keys );
			data_channel_access< nv_key_transform > kt_channel( nk_access.add_channel<nv_key_transform>( max_keys ) );

			raw_channel_interpolator interpolator( old_keys );
			data_descriptor final_key = interpolator.get_interpolation_key();

			for ( unsigned n = 0; n < max_keys; ++n )
			{
				float key[ 16 ];
				float* pkey = key;

				for ( uint16 c = 0; c < chan_count; ++c )
				{
					size_t idx = nv::min( old_keys->get_channel_size(c) - 1, n );
					pkey += raw_channel_interpolator::get_raw( *old_keys->get_channel(c), idx, pkey );
				}
				kt_channel.data()[n].tform = extract_key_raw< nv::transform >( final_key, key );
			}

			delete old_keys;
			m_data->m_nodes[i].data = new_keys;
		}
	}
}

void nv::mesh_nodes_creator::transform( float scale, const mat3& r33 )
{
	mat3 ri33 = glm::inverse( r33 );
	mat4 pre_transform ( scale * r33 );
	mat4 post_transform( 1.f/scale * ri33 ); 

	for ( size_t i = 0; i < m_data->get_count(); ++i )
	{
		mesh_node_data& node = m_data->m_nodes[i];
		node.transform = pre_transform * node.transform * post_transform;
		if ( node.data )
		{
			data_channel_set* kdata  = node.data;
			for ( size_t c = 0; c < kdata->size(); ++c )
			{
				raw_data_channel_access channel( kdata, c );
				size_t key_size = channel.element_size();
				for ( size_t n = 0; n < channel.size(); ++n )
				{
					transform_key_raw( kdata->get_channel( c )->descriptor(), channel.raw_data() + n * key_size, scale, r33, ri33 );
				}
			}
		}
	}
}

void nv::mesh_data_creator::transform( float scale, const mat3& r33 )
{
	vec3 vertex_offset     = vec3(); 
	mat3 vertex_transform  = scale * r33;
	mat3 normal_transform  = r33;

	for ( uint32 c = 0; c < m_data->size(); ++c )
	{
		raw_data_channel_access channel( m_data, c );
		const data_descriptor&  desc    = channel.descriptor();
		uint8* raw_data = channel.raw_data();
		uint32 vtx_size = desc.element_size();
		int p_offset = -1;
		int n_offset = -1;
		int t_offset = -1;
		for ( const auto& cslot : desc  )
			switch ( cslot.vslot )
			{
				case slot::POSITION : if ( cslot.etype == FLOAT_VECTOR_3 ) p_offset = int( cslot.offset ); break;
				case slot::NORMAL   : if ( cslot.etype == FLOAT_VECTOR_3 ) n_offset = int( cslot.offset ); break;
				case slot::TANGENT  : if ( cslot.etype == FLOAT_VECTOR_4 ) t_offset = int( cslot.offset ); break;
				default             : break;
			}

		if ( p_offset != -1 )
			for ( uint32 i = 0; i < channel.size(); i++)
			{
				vec3& p = *reinterpret_cast<vec3*>( raw_data + vtx_size*i + p_offset );
				p = vertex_transform * p + vertex_offset;
			}

		if ( n_offset != -1 )
			for ( uint32 i = 0; i < channel.size(); i++)
			{
				vec3& n = *reinterpret_cast<vec3*>( raw_data + vtx_size*i + n_offset );
				n = glm::normalize( normal_transform * n );
			}
		if ( t_offset != -1 )
			for ( uint32 i = 0; i < channel.size(); i++)
			{
				vec4& t = *reinterpret_cast<vec4*>(raw_data + vtx_size*i + t_offset );
				t = vec4( glm::normalize( normal_transform * vec3(t) ), t[3] );
			}
	}
}

struct vertex_g
{
	nv::vec4 tangent;
};

void nv::mesh_data_creator::flip_normals()
{
	int ch_n  = m_data->get_channel_index( slot::NORMAL );
	size_t n_offset = 0;
	if ( ch_n == -1 ) return;
	raw_data_channel_access channel( m_data, unsigned( ch_n ) );
	for ( const auto& cslot : channel.descriptor() )
		if ( cslot.vslot == slot::NORMAL )
		{
			n_offset  = cslot.offset;
		}

	for ( uint32 i = 0; i < channel.size(); ++i )
	{
		vec3& normal = *reinterpret_cast<vec3*>( channel.raw_data() + channel.element_size() * i + n_offset );
		normal = -normal;
	}
}


void nv::mesh_data_creator::generate_tangents()
{
	int p_offset = -1;
	int n_offset = -1;
	int t_offset = -1;
	datatype i_type = NONE;
	uint32 n_channel_index = 0;

	const raw_data_channel* p_channel = nullptr;
	      raw_data_channel* n_channel = nullptr;
	const raw_data_channel* t_channel = nullptr;
	const raw_data_channel* i_channel = nullptr;

	for ( uint32 c = 0; c < m_data->size(); ++c )
	{
		const raw_data_channel* channel = m_data->get_channel(c);

		for ( const auto& cslot : channel->descriptor() )
		switch ( cslot.vslot )
		{
			case slot::POSITION : 
				if ( cslot.etype == FLOAT_VECTOR_3 )
				{
					p_offset  = int( cslot.offset );
					p_channel = channel;
				}
				break;
			case slot::NORMAL   : 
				if ( cslot.etype == FLOAT_VECTOR_3 )
				{
					n_offset  = int( cslot.offset );
					n_channel = data_channel_set_creator( m_data )[ c ];
					n_channel_index = c;
				}
				break;
			case slot::TEXCOORD : 
				if ( cslot.etype == FLOAT_VECTOR_2 )
				{
					t_offset  = int( cslot.offset );
					t_channel = channel;
				}
				break;
			case slot::INDEX    : 
				{
					i_type    = cslot.etype;
					i_channel = channel;
				}
				break;
			case slot::TANGENT  : return;
			default             : break;
		}
	}
	if ( !p_channel || !n_channel || !t_channel ) return;

	if ( p_channel->size() != n_channel->size()
		|| p_channel->size() % t_channel->size() != 0
		|| ( i_type != UINT && i_type != USHORT && i_type != NONE ) )
	{
		return;
	}

	raw_data_channel g_channel  = data_channel_creator::create< vertex_g >( p_channel->size() );
	vec4* tangents              = &( data_channel_access< vertex_g >( &g_channel ).data()[0].tangent );
	vec3* tangents2             = new vec3[ p_channel->size() ];
	uint32 tri_count = i_channel ? i_channel->size() / 3 : t_channel->size() / 3;
	uint32 vtx_count = p_channel->size();
	uint32 sets      = p_channel->size() / t_channel->size();

	for ( unsigned int i = 0; i < tri_count; ++i )
	{
		uint32 ti0 = 0;
		uint32 ti1 = 0;
		uint32 ti2 = 0;
		if ( i_type == UINT )
		{
			const uint32* idata = reinterpret_cast<const uint32*>( i_channel->raw_data() );
			ti0 = idata[ i * 3 ];
			ti1 = idata[ i * 3 + 1 ];
			ti2 = idata[ i * 3 + 2 ];
		}
		else if ( i_type == USHORT )
		{
			const uint16* idata = reinterpret_cast<const uint16*>( i_channel->raw_data() );
			ti0 = idata[ i * 3 ];
			ti1 = idata[ i * 3 + 1 ];
			ti2 = idata[ i * 3 + 2 ];
		}
		else // if ( i_type == NONE )
		{
			ti0 = i * 3;
			ti1 = i * 3 + 1;
			ti2 = i * 3 + 2;
		}

		const vec2& w1 = *reinterpret_cast<const vec2*>(t_channel->raw_data() + t_channel->element_size()*ti0 + t_offset );
		const vec2& w2 = *reinterpret_cast<const vec2*>(t_channel->raw_data() + t_channel->element_size()*ti1 + t_offset );
		const vec2& w3 = *reinterpret_cast<const vec2*>(t_channel->raw_data() + t_channel->element_size()*ti2 + t_offset );
		vec2 st1 = w3 - w1;
		vec2 st2 = w2 - w1;
		float stst = (st1.x * st2.y - st2.x * st1.y);
		float coef = ( stst != 0.0f ? 1.0f / stst : 0.0f );

		for ( uint32 set = 0; set < sets; ++set )
		{
			uint32 nti0 = t_channel->size() * set + ti0;
			uint32 nti1 = t_channel->size() * set + ti1;
			uint32 nti2 = t_channel->size() * set + ti2;
			const vec3& v1 = *reinterpret_cast<const vec3*>(p_channel->raw_data() + p_channel->element_size()*nti0 + p_offset );
			const vec3& v2 = *reinterpret_cast<const vec3*>(p_channel->raw_data() + p_channel->element_size()*nti1 + p_offset );
			const vec3& v3 = *reinterpret_cast<const vec3*>(p_channel->raw_data() + p_channel->element_size()*nti2 + p_offset );
			vec3 xyz1 = v3 - v1;
			vec3 xyz2 = v2 - v1;

			//vec3 normal = glm::cross( xyz1, xyz2 );
			//
			//vtcs[ ti0 ].normal += normal;
			//vtcs[ ti1 ].normal += normal;
			//vtcs[ ti2 ].normal += normal;
			vec3 tangent  = (( xyz1 * st2.y ) - ( xyz2 * st1.y )) * coef;
			vec3 tangent2 = (( xyz2 * st1.x ) - ( xyz1 * st2.x )) * coef;

			tangents[nti0] = vec4( vec3( tangents[nti0] ) + tangent, 0 );
			tangents[nti1] = vec4( vec3( tangents[nti1] ) + tangent, 0 );
			tangents[nti2] = vec4( vec3( tangents[nti2] ) + tangent, 0 );

			tangents2[nti0] += tangent2;
			tangents2[nti1] += tangent2;
			tangents2[nti2] += tangent2;
		}
	}

	for ( unsigned int i = 0; i < vtx_count; ++i )
	{
		const vec3 n = *reinterpret_cast<const vec3*>( n_channel->raw_data() + n_channel->element_size()*i + n_offset );
		const vec3 t = vec3(tangents[i]);
		if ( ! ( t.x == 0.0f && t.y == 0.0f && t.z == 0.0f ) )
		{
			tangents[i]    = vec4( glm::normalize(t - n * glm::dot( n, t )), 0.0f ); 
			tangents[i][3] = (glm::dot(glm::cross(n, t), tangents2[i]) < 0.0f) ? -1.0f : 1.0f;
		}
	}
	delete[] tangents2;

	data_channel_set_creator( m_data ).set_channel( n_channel_index, merge_channels( *n_channel, g_channel ) );
}

nv::raw_data_channel nv::mesh_data_creator::merge_channels( const raw_data_channel& a, const raw_data_channel& b )
{
	NV_ASSERT( a.size() == b.size(), "merge_channel - bad channels!" );
	data_descriptor desc  = a.descriptor();
	desc.append( b.descriptor() );

	raw_data_channel result = data_channel_creator::create( desc, a.size() );
	for ( uint32 i = 0; i < a.size(); ++i )
	{
		raw_copy_n( a.raw_data() + i * a.element_size(), a.element_size(), raw_data_channel_access( &result ).raw_data() + i*desc.element_size() );
		raw_copy_n( b.raw_data() + i * b.element_size(), b.element_size(), raw_data_channel_access( &result ).raw_data() + i*desc.element_size() + a.element_size() );
	}

	return result;
}

nv::raw_data_channel nv::mesh_data_creator::append_channels( const raw_data_channel& a, const raw_data_channel& b, uint32 frame_count )
{
	NV_ASSERT( a.descriptor() == b.descriptor(), "Merge - append not compatible format!" );
	NV_ASSERT( a.size() % frame_count == 0, "Merge - append first mesh empty!" );
	NV_ASSERT( b.size() % frame_count == 0, "Merge - append second mesh empty!" );
	size_t vtx_size = a.element_size();

	raw_data_channel result = data_channel_creator::create( a.descriptor(), a.size() + b.size() );
	uint8* rdata = raw_data_channel_access( &result ).raw_data();

	if ( frame_count == 1 )
	{
		size_t a_size = vtx_size * a.size();
		raw_copy_n( a.raw_data(), a_size, rdata );
		raw_copy_n( b.raw_data(), vtx_size * b.size(), rdata + a_size );
	}
	else
	{
		size_t frame_size_a = ( a.size() / frame_count ) * vtx_size;
		size_t frame_size_b = ( b.size() / frame_count ) * vtx_size;
		size_t pos_a = 0;
		size_t pos_b = 0;
		size_t pos   = 0;
		for ( size_t i = 0; i < frame_count; ++i )
		{
			raw_copy_n( a.raw_data() + pos_a, frame_size_a, rdata + pos );
			raw_copy_n( b.raw_data() + pos_b, frame_size_b, rdata + pos + frame_size_a );				pos_a += frame_size_a;
			pos_b += frame_size_b; 
			pos   += frame_size_a + frame_size_b;
		}
	}

	return result;
}



bool nv::mesh_data_creator::is_same_format( mesh_data* other )
{
	if ( m_data->size() != other->size() ) return false;
	for ( uint32 c = 0; c < m_data->size(); ++c )
	{
		if ( m_data->get_channel(c)->descriptor() != other->get_channel(c)->descriptor() )
			return false;
	}
	return true;
}

void nv::mesh_data_creator::merge( mesh_data* other )
{
	if ( !is_same_format( other ) ) return;
	int ch_pi  = m_data->get_channel_index( slot::POSITION );
	int ch_ti  = m_data->get_channel_index( slot::TEXCOORD );
	int och_pi = other->get_channel_index( slot::POSITION );
	int och_ti = other->get_channel_index( slot::TEXCOORD );
	if ( ch_pi == -1 || ch_ti == -1 ) return;
	size_t size   = m_data->get_channel_size( unsigned(ch_ti) );
	size_t osize  =  other->get_channel_size( unsigned(och_ti) );
	size_t count  = m_data->get_channel_size( unsigned(ch_pi) );
	size_t ocount =  other->get_channel_size( unsigned(och_pi) );
	if ( count % size != 0 || ocount % osize != 0 ) return;
	if ( count / size != ocount / osize ) return;
	
	data_channel_set_creator data( m_data );

	for ( uint32 c = 0; c < m_data->size(); ++c )
	{
		const raw_data_channel* old = m_data->get_channel( c );
		uint32 old_size = old->size();
		data_descriptor old_desc = old->descriptor();
		bool old_is_index = old_size > 0 && old_desc[0].vslot == slot::INDEX;
		size_t frame_count = ( old_is_index ? 1 : old_size / size );
		data.set_channel( c, append_channels( *old, *other->get_channel(c), frame_count ) );
		if ( old_is_index )
		{
			switch ( old_desc[0].etype )
			{
			case USHORT : 
				{
					NV_ASSERT( size + osize < uint16(-1), "Index out of range!" );
					raw_data_channel_access ic( data[c] );
					uint16* indexes = reinterpret_cast<uint16*>( ic.raw_data() );
					for ( uint16 i = uint16( old_size ); i < ic.size(); ++i )
						indexes[i] += uint16( size );

				}
				break;
			case UINT   : 
				{
					raw_data_channel_access ic( data[c] );
					uint32* indexes = reinterpret_cast<uint32*>( ic.raw_data() );
					for ( uint32 i = old_size; i < ic.size(); ++i )
						indexes[i] += size;
				}
				break;
			default : NV_ASSERT( false, "Unsupported index type!" ); break;
			}
		}
	}
}

void nv::mesh_creator::delete_mesh( uint32 index )
{
	if ( index < m_pack->get_count() )
	{
		data_channel_set_creator( &m_pack->m_meshes[m_pack->m_count - 1] ).move_to( m_pack->m_meshes[index] );
		m_pack->m_count--;
	}
}
