// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file allocator.hh
 * @author Kornel Kisielewicz
 * @brief allocator type
 *
 * Allocator concept based on:
 *
 * http://bitsquid.blogspot.com/2010/09/custom-memory-allocation-in-c.html
 * http://www.gamedev.net/page/resources/_/technical/general-programming/c-custom-memory-allocation-r3010
 */

#ifndef NV_STL_ALLOCATOR_HH
#define NV_STL_ALLOCATOR_HH

#include <nv/common.hh>

namespace nv
{

	class allocator
	{
	public:
		virtual void* allocate( size_t size, size_t alignment ) = 0;
		virtual void deallocate( void *p ) = 0;
		virtual void* reallocate( void* p, size_t new_size ) = 0;
		virtual size_t allocated_size( void *p ) = 0;

		virtual size_t get_allocated_count() = 0;
		virtual size_t get_allocated_size() = 0;
		virtual size_t get_max_allocated_count() = 0;
		virtual size_t get_max_allocated_size() = 0;

		template < class T, typename... Args > 
		T *create( Args&&... args ) 
		{
			return new ( allocate( sizeof( T ), alignof( T ) ) ) T( forward<Args>( args )... );
		}

		template <class T> 
		void destroy( T *p )
		{
			if ( p )
			{
				p->~T();
				deallocate( p );
			}
		}
	};



}

#endif // NV_CORE_ALLOCATOR_HH
