// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file growing_container_handler.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief growing contiguous container handler
*/

#ifndef NV_STL_CONTAINER_GROWING_CONTAINER_HANDLER_HH
#define NV_STL_CONTAINER_GROWING_CONTAINER_HANDLER_HH

#include <nv/common.hh>
#include <nv/stl/container/initialize_policy.hh>
#include <nv/stl/container/sized_container_handler.hh>

namespace nv
{

	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard
	>
	class growing_container_handler : public sized_container_handler< Storage, InitializePolicy >
	{
	public:
		typedef typename growing_container_handler< Storage, InitializePolicy > this_type;
		typedef typename Storage::value_type value_type;
		typedef typename Storage::size_type  size_type;
		typedef value_type*                  iterator;
		typedef const value_type*            const_iterator;

		using sized_container_handler< Storage, InitializePolicy >::sized_container_handler;

		void reserve( size_type new_capacity )
		{
			Storage::try_reserve( new_capacity, true );
		}
		void push_back( const value_type& e )
		{
			if ( Storage::try_grow( 1 ) ) copy_construct_object( Storage::data() + Storage::size() - 1, e );
		}
		void push_back( value_type&& e )
		{
			if ( Storage::try_grow( 1 ) ) move_construct_object( Storage::data() + Storage::size() - 1, forward<value_type>( e ) );
		}
		template < typename... Args >
		void emplace_back( Args&&... args )
		{
			if ( Storage::try_grow( 1 ) ) construct_object( Storage::data() + Storage::size() - 1, forward<Args>( args )... );
		}

		void pop_back()
		{
			InitializePolicy::destroy( Storage::data() + Storage::size() - 1 );
			Storage::try_resize( Storage::size() - 1, true );
		}

		// TODO: implement swap_erase
		iterator erase( iterator position )
		{
			iterator iend = Storage::data() + Storage::size();
			InitializePolicy::destroy( position );
			if ( ( position + 1 ) < iend )
				raw_alias_copy( position + 1, iend, position );
			Storage::try_resize( Storage::size() - 1, true );
			return position;
		}

		iterator erase( iterator first, iterator last )
		{
			iterator iend = Storage::data() + Storage::size();
			InitializePolicy::destroy( first, last );
			iterator position = raw_alias_copy( last, iend, first );
			Storage::try_resize( Storage::size() - ( last - first ), true );
			return position;
		}

		void append( const_iterator first, const_iterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			size_type d        = distance( first, last );
			size_type old_size = Storage::size();
			if ( Storage::try_grow( d ) )
				InitializePolicy::copy( first, last, Storage::data() + old_size );
		}

		// TODO: This can be much optimized in the grow case by copying in three parts
		void insert( iterator position, const value_type& value )
		{
			iterator iend = Storage::data() + Storage::size();
			if ( Storage::try_grow( 1 ) )
			{
				raw_alias_copy( position, iend, position + 1 );
				copy_construct_object( position, value );
			}
		}

		// TODO: This can be much optimized in the grow case by copying in three parts
		void insert( iterator position, const_iterator first, const_iterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			iterator iend = Storage::data() + Storage::size();
			size_type d = distance( first, last );
			if ( Storage::try_grow( d ) )
			{
				raw_alias_copy( position, iend, position + d );
				InitializePolicy::copy( first, last, position );
			}
		}
	};

}

#endif // #define NV_STL_CONTAINER_GROWING_CONTAINER_HANDLER_HH

