// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file flags.hh
 * @author Kornel Kisielewicz
 * @brief flags
 */

#ifndef NV_STL_FLAGS_HH
#define NV_STL_FLAGS_HH

#include <nv/common.hh>
#include <nv/stl/type_traits/transforms.hh>
#include <nv/stl/algorithm.hh>

namespace nv
{
	template < uint32 SIZE, typename T = uint32 >
	class flags
	{
	public:
		class reference
		{
			friend class flags<SIZE,T>;
		public:
			typedef bool value_type;
			typedef T    index_type;

			reference& operator=( value_type a_value )
			{
				m_flags->set( m_index, a_value );
				return (*this);
			}
			reference& operator=( const reference& a_value )
			{
				m_flags->set( m_index, bool( a_value ) );
				return (*this);
			}
			operator bool() const 
			{
				return m_flags->test( m_index );
			}

		private:
			reference() : m_flags( nullptr ), m_index( index_type(0) ) {}

			reference( flags<SIZE,T>* a_flags, index_type a_index )
				: m_flags( a_flags ), m_index( a_index )
			{	
			}

		private:
			flags<SIZE,T>* m_flags;
			index_type     m_index;
		};

		class enumerator
		{
			friend class flags<SIZE,T>;
		public:
			typedef T         index_type;
			typedef T         value_type;
			typedef T*        iterator;
			typedef const T*  const_iterator;
			typedef T&        reference;
			typedef const T&  const_reference;
			typedef size_t    size_type;
			typedef ptrdiff_t difference_type;

			T operator* () const { return m_index; }
			T const* operator-> () const { return &m_index; }
			bool operator== ( const enumerator& rhs ) const
			{
				return ( m_index == rhs.m_index ) && ( m_flags == rhs.m_flags );
			}
			bool operator!= ( const enumerator& rhs ) const
			{
				return !( *this == rhs );
			}
			
			enumerator& operator++ () 
			{
				next();
				return *this; 
			}
			enumerator operator++ (int) 
			{
				auto result = *this; 
				++*this; 
				return result;
			}
		protected:
			enumerator() : m_flags( nullptr ), m_index( index_type(0) ) {}

			enumerator( const flags<SIZE,T>* a_flags, index_type a_index )
				: m_flags( a_flags ), m_index( a_index )
			{	
				if ( a_flags && !a_flags->test( a_index ) ) next();
			}

			void next()
			{
				if ( m_flags )
				do 
				{
					if ( raw_index_type(m_index) >= SIZE ) { m_flags = nullptr; m_index = index_type(0); return; }
					m_index = index_type((raw_index_type)m_index + 1);
				} while ( !m_flags->test( m_index ) );
			}

			const flags<SIZE,T>* m_flags;
			index_type           m_index;
		};

		typedef uint8     data_type;
		typedef T         index_type;
		typedef uint32 size_type;
		typedef make_underlying_type_t<T> raw_index_type;

		static const size_type data_type_size = sizeof( data_type ) * 8;
		static const size_type data_count     = SIZE;
		static const size_type data_size      = (SIZE+data_type_size-1) / data_type_size;

		enumerator begin()  const { return enumerator( this, index_type(0) ); }
		enumerator cbegin() const { return enumerator( this, index_type(0) ); }

		enumerator end()  const { return enumerator(); }
		enumerator cend() const { return enumerator(); }

		flags()
		{
			reset();
		}

		explicit flags( const data_type* a_data )
		{
			assign( a_data );
		}

		void assign( const data_type* a_data )
		{
			raw_copy_n( a_data, data_size, m_data );
		}

		void reset()
		{
			raw_zero_n( m_data, data_size );
		}

		void include( index_type i )
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			m_data[ idx ] |= 1 << static_cast< data_type >( pos );
		}

		void exclude( index_type i )
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			m_data[ idx ] &= ~(1 << static_cast< data_type >( pos ) );
		}

		void set( index_type i, bool value )
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			if ( value )
				m_data[ idx ] |= 1 << static_cast< data_type >( pos );
			else
				m_data[ idx ] &= ~(1 << static_cast< data_type >( pos ) );
		}

		bool test( index_type i ) const
		{
			raw_index_type idx = static_cast< raw_index_type >( i ) / data_type_size;
			raw_index_type pos = static_cast< raw_index_type >( i ) % data_type_size;
			return ( m_data[ idx ] & ( 1 << static_cast< data_type >( pos ) ) ) != 0;
		}

		const data_type* data() const
		{
			return m_data;
		}

		data_type* data()
		{
			return m_data;
		}

		size_type size() const
		{
			return data_count;
		}

		size_type capacity() const
		{
			return data_size;
		}

		bool operator[]( index_type idx ) const
		{
			return test( idx );
		}

		reference operator[]( index_type idx )
		{
			return reference( this, idx );
		}

	private:
		data_type m_data[ data_size ];
	};

} // namespace nv

#endif // NV_STL_FLAGS_HH
