// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.
//
// TODO: decouple from the type database
// TODO: paths for function registration?


#ifndef NV_LUA_STATE_HH
#define NV_LUA_STATE_HH

#include <nv/common.hh>
#include <nv/stl/flags.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/type_traits/function.hh>

#include <nv/lua/lua_handle.hh>
#include <nv/lua/lua_path.hh>
#include <nv/lua/lua_values.hh>
#include <nv/lua/lua_dispatch.hh>


#ifdef NV_DEBUG
#define NV_LUA_STACK_ASSERT( state, value ) nv::lua::stack_assert __lua_stack_assert( (state), (value) );
#else
#define NV_LUA_STACK_ASSERT( (state), (value) )
#endif

namespace nv
{
	namespace lua
	{
		class state;

		const int ret_multi = -1;

		class state_wrapper
		{
		public:
			state_wrapper( lua_State* a_state, bool a_owner ) : m_state( a_state ), m_owner( a_owner ), m_global( true ) {}
			virtual ~state_wrapper();

			template < typename R >
			R call( const path& p )
			{
				if ( push_function( p, m_global ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename... Args >
			R call(const path& p, Args&&... args )
			{
				if (push_function(p, m_global))
				{
					detail::push_values(m_state, std::forward<Args>(args)...);
					if (call_function(sizeof...(Args), detail::return_count< R >::value) == 0)
					{
						return detail::pop_return_value<R>(m_state);
					}
				}
				return R();
			}

			template< typename R >
			R get( const path& p )
			{
				if ( p.resolve( m_state, m_global ) )
				{
					return detail::pop_return_value<R>( m_state );
				}
				return R();
			}

			template< typename R, typename T >
			R get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			template< typename R, typename T >
			R raw_get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get_raw();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			template< typename R >
			R get( const path& p, const R& def )
			{
				if ( p.resolve( m_state, m_global ) )
				{
					return detail::pop_return_value<R>( m_state, def );
				}
				return R();
			}

			template< typename R, typename T >
			R get( const T& key, const R& def )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state, def );
			}

			template< typename R, typename T >
			R raw_get( const T& key, const R& def )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get_raw();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state, def );
			}
			bool is_defined( const path& p ) { return is_defined( p, m_global ); }
			void register_native_function( lfunction f, string_view name );

			template < typename F, F f >
			void register_function( string_view name )
			{
				register_native_function( detail::function_wrapper< F, f >, name );
			}

			template < typename C, typename F, F f >
			void register_function( string_view name )
			{
				register_native_function( detail::object_method_wrapper< C, F, f >, name );
			}

		protected:
			bool is_defined( const path& p, bool global );
			bool push_function( const path& p, bool global );
			int call_function( int nargs, int nresults );
		private:
			void push_global_table();
			void pop_global_table();
			void call_get();
			void call_get_raw();
		protected:
			lua_State* m_state;
			bool m_owner;
			bool m_global;
		};

		class stack_guard
		{
		public:
			stack_guard( state* aL );
			stack_guard( state& aL );
			int get_level() const { return m_level; }
			~stack_guard();
		private:
			state* L;
			int m_level;
		};

		class stack_assert
		{
		public:
			stack_assert( state* aL, int expected );
			stack_assert( state& aL, int expected );
			stack_assert( lua_State* sL, int expected );
			~stack_assert();
		private:
			lua_State* L;
			int m_expected;
		};


		class table_guard;
		class state : public state_wrapper
		{
			friend class stack_guard;
			friend class table_guard;

		public:
			explicit state( bool load_libs = false );
			explicit state( lua_State* state );
			bool do_string( string_view code, string_view name, int rvalues = 0 );
			bool do_file( string_view filename );
			int get_stack_size();
			void log_stack();
			lua_State* get_raw();
			ref register_object( void* o, string_view lua_name );
			ref register_proto( string_view id, string_view storage );
			void store_metadata( ref object_index, string_view metaname, void* pointer );
			void unregister_object( ref object_index );

			void register_enum( string_view name, int value );
			void register_singleton( string_view name, void* o );

			void register_native_object_method( string_view lua_name, string_view name, lfunction f );
			template < typename F, F f >
			void register_object_method( string_view lua_name, string_view name )
			{
				register_native_object_method( lua_name, name, detail::object_method_wrapper< typename memfn_class_type<F>::type, F, f > );
			}
			operator lua_State*() { return m_state; }

			template < typename H >
			void register_handle( const H& handle )
			{
				nv::lua::register_handle( m_state, handle, true );
			}
			template < typename H >
			void unregister_handle( const H& handle )
			{
				nv::lua::unregister_handle( m_state, handle );
			}
			template < typename H >
			ref register_handle_component( const H& handle, string_view id )
			{
				nv::lua::push_handle( m_state, handle );
				return register_handle_component_impl( id, true );
			}
			template < typename H >
			ref register_handle_component( const H& handle, string_view id, const path& proto )
			{
				if ( !proto.resolve( m_state, true ) )
				{
					return ref();
				}
				nv::lua::push_handle( m_state, handle );
				return register_handle_component_impl( id, false );
			}
			template < typename H >
			void unregister_handle_component( const H& handle, string_view id )
			{
				nv::lua::push_handle( m_state, handle );
				unregister_handle_component_impl( id );
			}

			template < typename R >
			R call( ref table, const path& p )
			{
				stack_guard guard( this ); // TODO: remove
				detail::push_ref_object( m_state,table );
				if ( push_function( p, false ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename... Args >
			R call( ref table, const path& p, Args&&... args )
			{
				stack_guard guard( this ); // TODO: remove
				detail::push_ref_object( m_state,table );
				if ( push_function( p, false ) )
				{
					detail::push_values( m_state, std::forward<Args>(args)... );
					if (call_function(sizeof...(Args), detail::return_count< R >::value) == 0)
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}

		private:
			ref register_handle_component_impl( string_view id, bool empty );
			void unregister_handle_component_impl( string_view id );

			int load_string( string_view code, string_view name );
			int load_file( string_view filename );
			int do_current( string_view name, int rvalues = 0 );
			void deep_pointer_copy( int index, void* obj );
		};

		class table_guard : public state_wrapper
		{
		public:
			table_guard( state* lstate, const path& p, bool global = true );
			table_guard( const table_guard& parent, const path& p );
			virtual ~table_guard();
			size_t get_size();
			bool has_field( string_view element );
			std::string get_std_string( string_view element, string_view defval = string_view() );
			const_string get_string( string_view element, string_view defval = string_view() );
			char get_char( string_view element, char defval = ' ' );
			int get_integer( string_view element, int defval = 0 );
			unsigned get_unsigned( string_view element, unsigned defval = 0 );
			double get_double( string_view element, double defval = 0.0 );
			float get_float( string_view element, float defval = 0.0 );
			bool get_boolean( string_view element, bool defval = false );
			bool is_table( string_view element );
			bool is_number( string_view element );
			bool is_boolean( string_view element );
			bool is_string( string_view element );
		private:
			int m_level;
		};



	} // namespace lua

} // namespace nv

#endif // NV_LUA_STATE_HH
