// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file alignment.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief type traits - alignment
 */

#ifndef NV_STL_TRAITS_ALIGNMENT_HH
#define NV_STL_TRAITS_ALIGNMENT_HH

#include <nv/stl/type_traits/common.hh>

namespace nv
{

	namespace detail
	{
		class aligned_dummy;
		typedef void( *aligned_fptr )( );
		typedef int( aligned_dummy::*aligned_memptr );
		typedef int ( aligned_dummy::*aligned_memfptr )( );
	}

#if NV_COMPILER == NV_MSVC 
#pragma warning( push )
#pragma warning( disable : 4121 ) 
#endif		
	union max_align_t
	{
		double                  dummy0;
		long double             dummy1;
		void*                   dummy2;
		long long               dummy3;
		detail::aligned_fptr    dummy4;
		detail::aligned_memptr  dummy5;
		detail::aligned_memfptr dummy6;
	};
#if NV_COMPILER == NV_MSVC 
#pragma warning( pop )
#endif		

#if NV_COMPILER == NV_CLANG
	template< typename T, size_t Size, size_t Align >
	struct aligned_array
	{
		typedef T alignas( Align ) type[Size];
	};
#elif NV_COMPILER == NV_GNUC
	template< typename T, size_t Size, size_t Align >
	struct aligned_array
	{
		typedef T __attribute__( ( aligned( Align ) ) ) type[Size];
	};
#else
	// TODO: remove this shit after moving to MSVC 2015
	template< typename T, size_t Size, size_t Align >
	struct aligned_array;

	// According to LLVM ( https://llvm.org/svn/llvm-project/llvm/trunk/include/llvm/Support/AlignOf.h )
	// MSVC has problems with align below 16...
#define NV_ALIGNED_ARRAY(N) \
	template< typename T, size_t Size > \
	struct aligned_array< T, Size, N >  \
			{ \
		typedef __declspec( align(N) ) T type[Size]; \
			}; 

	NV_ALIGNED_ARRAY( 1 )
		NV_ALIGNED_ARRAY( 2 )
		NV_ALIGNED_ARRAY( 4 )
		NV_ALIGNED_ARRAY( 8 )
		NV_ALIGNED_ARRAY( 16 )
		NV_ALIGNED_ARRAY( 32 )
		NV_ALIGNED_ARRAY( 64 )
		NV_ALIGNED_ARRAY( 128 )

#undef NV_ALIGNED_ARRAY
#endif
	
	template< typename T, size_t Size, size_t Align >
	using aligned_array_t = typename aligned_array< T, Size, Align >::type;

	template< size_t Size, size_t Align = alignof( max_align_t ) >
	struct aligned_storage
	{
		struct type
		{
			typename aligned_array< unsigned char, Size, Align >::type data;
		};
	};

	template< size_t Size, size_t Align = alignof( max_align_t ) >
	using aligned_storage_t = typename aligned_storage<Size, Align>::type;

	namespace detail
	{
		template < size_t... Sizes >
		struct size_max;

		template <>
		struct size_max < >
		{
			static constexpr size_t value = 0;
		};

		template < size_t Size >
		struct size_max < Size >
		{
			static constexpr size_t value = Size;
		};

		template < size_t S1, size_t S2, size_t... Sizes >
		struct size_max< S1, S2, Sizes... > : size_max < ( S1 < S2 ? S2 : S1 ), Sizes... > {};
	}

	template< size_t Size, typename... Types >
	struct aligned_union
	{
		static constexpr size_t max_length = detail::size_max < Size, sizeof( Types )... >::value;
		static constexpr size_t alignment_value = detail::size_max < alignment_of< Types >::value... >::value;
		struct type
		{
			typename aligned_array< unsigned char, max_length, alignment_value >::type data;
		};
	};

	template < size_t Size, typename... Types >
	using aligned_union_t = typename aligned_union<Size, Types...>::type;

}

#endif // NV_STL_TRAITS_ALIGNMENT_HH
