// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file hash_table_policy.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief hash table policy classes
*/

#ifndef NV_STL_HASH_TABLE_POLICY_HH
#define NV_STL_HASH_TABLE_POLICY_HH

#include <nv/common.hh>
#include <nv/stl/functional/hash.hh>
#include <nv/stl/functional/comparisons.hh>
#include <nv/stl/memory.hh>
#include <nv/stl/utility/pair.hh>

namespace nv
{


	template <
		typename T,
		typename IsEqual = equal_to<T>,
		bool IsEmpty = is_empty< IsEqual >::value
	>
	struct compare_policy;


	template < typename T >
	struct compare_policy< T, equal_to<T>, true >
	{
		constexpr bool compare( const T& t1, const T& t2 ) const
		{
			return t1 == t2;
		}
		equal_to<T> key_eq() const { return equal_to<T>(); }
	};

	template < typename T, typename IsEqual >
	struct compare_policy< T, IsEqual, true > : private IsEqual
	{
		constexpr bool compare( const T& t1, const T& t2 ) const
		{
			return (*this)( t1 == t2 );
		}
		IsEqual key_eq() const { return ( *this ); }
	};

	template < typename T, typename IsEqual >
	struct compare_policy< T, IsEqual, false >
	{
		constexpr bool compare( const T& t1, const T& t2 ) const
		{
			return m_key_eq( t1 == t2 );
		}
		IsEqual key_eq() const { return m_key_eq; }
	private:
		IsEqual m_key_eq;
	};

	template <
		typename T,
		typename H,
		typename Hasher = hash< T, H >,
		bool IsEmpty = is_empty< Hasher >::value
	>
	struct hash_policy;


	template < typename T, typename H >
	struct hash_policy< T, H, hash< T, H >, true >
	{
		constexpr H get_hash( const T& t ) const
		{
			return hash< T, H >::get( t );
		}
		hash< T, H > hash_function() const { return hash< T, H >(); }
	};

	template < typename T, typename H, typename Hasher >
	struct hash_policy< T, H, Hasher, true > : private Hasher
	{
		constexpr H get_hash( const T& t ) const
		{
			return ( *this )( t );
		}
		hash< T, H > hash_function() const { return ( *this ); }
	};

	template < typename T, typename H, typename Hasher >
	struct hash_policy< T, H, Hasher, false >
	{
		constexpr H get_hash( const T& t ) const
		{
			return m_hasher( t );
		}
		hash< T, H > hash_function() const { return m_hasher; }
	private:
		Hasher m_hasher;
	};

	// Due to MSVC not being able to handle multiple empty bases, if both
	// hash and compare are empty, we will still get a 4-size structure.
	// We will optimize here, to assure that if either is empty, we will still
	// get an empty class
	template <
		typename T,
		typename H,
		typename IsEqual = equal_to< T >,
		typename Hasher = hash< T, H >,
		bool IsEqualIsEmpty = is_empty< IsEqual >::value,
		bool HasherIsEmpty = is_empty< Hasher >::value
	>
	struct hash_compare_policy;

	template <
		typename T,
		typename H,
		typename IsEqual,
		typename Hasher,
		bool HasherIsEmpty
	>
	struct hash_compare_policy< T, H, IsEqual, Hasher, false, HasherIsEmpty >
		: hash_policy< T, H, Hasher, HasherIsEmpty >
	{
		constexpr bool compare( const T& t1, const T& t2 ) const
		{
			return m_key_eq( t1 == t2 );
		}
		IsEqual key_eq() const { return m_key_eq; }
	private:
		IsEqual m_key_eq;
	};

	template <
		typename T,
		typename H,
		typename IsEqual,
		typename Hasher
	>
	struct hash_compare_policy< T, H, IsEqual, Hasher, true, false >
		: compare_policy< T, IsEqual, true >
	{
		constexpr H get_hash( const T& t ) const
		{
			return m_hasher( t );
		}
		hash< T, H > hash_function() const { return m_hasher; }
	private:
		Hasher m_hasher;
	};

	template <
		typename T,
		typename H,
		typename Hasher,
		typename IsEqual
	>
	struct hash_compare_policy< T, H, IsEqual, Hasher, true, true >
		: hash_policy< T, H, Hasher, true >
	{
		constexpr bool compare( const T& t1, const T& t2 ) const
		{
			static_assert( is_trivial< IsEqual >::value, "There is something really wrong with your equality functor." );
			return IsEqual()( t1, t2 );
		}
		IsEqual key_eq() const { return IsEqual(); }
	};

	template <
		typename Key,
		typename Value,
		typename Hash,
		typename IsEqual = equal_to< Key >,
		typename Hasher = hash< Key, Hash >
	>
	struct hash_table_entry_stl_map : hash_compare_policy< Key, Hash, IsEqual, Hasher > 
	{
		typedef hash_compare_policy< Key, Hash, IsEqual, Hasher > base_type;
		typedef pair< const Key, Value > value_type;
		typedef Key        key_type;
		typedef Hash       hash_type;
		typedef Value      mapped_type;

		struct entry_type
		{
			value_type value;
		};

		constexpr bool entry_compare( const entry_type* entry, const key_type& key, hash_type )
		{
			return base_type::compare( entry->value.first, key );
		}

		constexpr bool entry_compare( const entry_type* entry, const value_type& value, hash_type )
		{
			return base_type::compare( entry->value.first, value.first );
		}

		inline void entry_construct( entry_type* entry, const value_type& value, hash_type )
		{
			copy_construct_object( &(entry->value), value );
		}

		constexpr hash_type get_entry_hash( const entry_type* entry )
		{
			return base_type::get_hash( entry->value.first );
		}
		constexpr hash_type get_value_hash( const value_type& value )
		{
			return base_type::get_hash( value.first );
		}
	};

	template <
		typename Key,
		typename Value,
		typename Hash,
		typename IsEqual = equal_to< Key >,
		typename Hasher = hash< Key, Hash >
	>
	struct hash_table_entry_stl_map_with_hash : hash_compare_policy< Key, Hash, IsEqual, Hasher >
	{
		typedef hash_compare_policy< Key, Hash, IsEqual, Hasher > base_type;
		typedef pair< const Key, Value > value_type;
		typedef Key        key_type;
		typedef Hash       hash_type;
		typedef Value      mapped_type;

		struct entry_type
		{
			value_type value;
			hash_type  hash_code;
		};

		constexpr bool entry_compare( const entry_type* entry, const key_type& key, hash_type h )
		{
			return entry->hash_code == h && base_type::compare( entry->value.first, key );
		}
		constexpr bool entry_compare( const entry_type* entry, const value_type& value, hash_type h )
		{
			return entry->hash_code == h &&  base_type::compare( entry->value.first, value.first );
		}

		constexpr hash_type get_entry_hash( const entry_type* entry )
		{
			return entry->hash_code;
		}
		constexpr hash_type get_value_hash( const value_type& value )
		{
			return base_type::get_hash( value.first );
		}
		inline void construct( entry_type* entry, const value_type& value, hash_type hash_code )
		{
			copy_construct_object( &(entry->value), value );
			entry.hash_code = hash_code;
		}
	};
	
	struct hash_table_range_mod_policy
	{
		template< typename H >
		static size_t get( H h, size_t n ) { return h % n; }
	};

	uint32 get_prime_larger_or_equal_to( uint32 value );

	struct hash_table_prime_rehash_policy
	{
		static uint32 get_bucket_count( uint32 requested_count )
		{
			return get_prime_larger_or_equal_to( requested_count );
		}
		static uint32 get_bucket_count( uint32 element_count, float load_factor )
		{
			uint32 min_count = (uint32)( element_count / load_factor );
			return get_prime_larger_or_equal_to( min_count );
		}
		static uint32 is_rehash_required( uint32 bucket_count, uint32 element_count, float load_factor )
		{
			uint32 min_count = (uint32)( element_count / load_factor );
			if ( bucket_count < 2 || min_count > bucket_count )
			{
				return get_prime_larger_or_equal_to( min_count );
			}
			return 0;
		}
	};

}

#endif // NV_STL_HASH_TABLE_POLICY_HH
