// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file profiler.hh
 * @author Kornel Kisielewicz
 * @brief profiler
 */

#ifndef NV_CORE_PROFILER_HH
#define NV_CORE_PROFILER_HH

#include <nv/core/common.hh>
#include <nv/stl/singleton.hh>
#include <nv/stl/string.hh>
#include <nv/stl/unordered_map.hh>

#if NV_PROFILER 
#define NV_PROFILE( tag ) nv::profiler_guard __profile( tag )
#define NV_PROFILE_IF( tag, condition ) nv::profiler_condition_guard __profile( tag, condition )
#else
#define NV_PROFILE( tag )
#define NV_PROFILE_IF( tag, condition ) 
#endif


namespace nv
{
	class profiler : public auto_singleton< profiler >
	{
	protected:
		class node
		{
		public:
			friend class profiler;
			node* get_parent() { return m_parent; }
			node* get_child( const std::string& tag )
			{
				auto it = m_children.find( tag );
				return ( it != m_children.end() ) ? it->second : nullptr;
			}
		protected:
			node( const string_ref& tag, node* parent );
			node* request_child( const string_ref& tag );
			void start();
			bool stop();
			~node();
		protected:
			typedef unordered_map< std::string, node* > map;

			std::string m_tag;
			map         m_children;
			node*       m_parent;
			uint32      m_recusion;

			uint32      m_calls;
			uint64      m_start_time_us;
			uint64      m_total_time_us;
		};

	protected:
		profiler();
		~profiler();

		void start_profile( const string_ref& tag );
		void stop_profile();
	public:
		friend class auto_singleton< profiler >;
		friend class profiler_guard;
		friend class profiler_condition_guard;
		void log_report();
	private:
		void log_node_children( uint32 indent, const node* n );
		node* m_root;
		node* m_current;
	};

	class profiler_guard
	{
	public:
		profiler_guard( string_ref tag )
		{
			profiler::pointer()->start_profile( tag );
		}

		~profiler_guard()
		{
			profiler::pointer()->stop_profile();
		}
	};

	class profiler_condition_guard
	{
	public:
		profiler_condition_guard( string_ref tag, bool condition )
			: m_active( condition )
		{
			if ( m_active ) profiler::pointer()->start_profile( tag );
		}

		~profiler_condition_guard()
		{
			if ( m_active ) profiler::pointer()->stop_profile();
		}
	private:
		bool m_active;
	};

} // namespace nv

#endif // NV_CORE_PROFILER_HH
