// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
/**
* @file properties.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief type traits - type properties
*/

#ifndef NV_STL_TRAITS_PROPERTIES_HH
#define NV_STL_TRAITS_PROPERTIES_HH

#include <nv/core/common.hh>
#include <nv/stl/type_traits/common.hh>
#include <nv/stl/type_traits/primary.hh>

namespace nv
{
	template < typename T >
	struct is_fundamental : bool_constant <
		is_void<T>::value || 
		is_nullptr<T>::value ||
		is_integral<T>::value ||
		is_floating_point<T>::value
	> {};

	template < typename T >
	struct is_arithmetic : bool_constant < 
		is_integral<T>::value || 
		is_floating_point<T>::value 
	> {};

	template < typename T >
	struct is_scalar : bool_constant <
		is_arithmetic<T>::value ||
		is_enum<T>::value ||
		is_pointer<T>::value ||
		is_member_pointer<T>::value ||
		is_nullptr<T>::value
	> {};

	template < typename T >
	struct is_object : bool_constant <
		is_scalar<T>::value ||
		is_array<T>::value ||
		is_union<T>::value ||
		is_class<T>::value
	> {};

	template < typename T >
	struct is_compound : bool_constant < !is_fundamental< T >::value > {};

	// TODO : confirm conformance
#if NV_COMPILER == NV_MSVC
	template < typename T >	struct is_trivially_copyable : bool_constant < ( __has_trivial_copy( T ) || __is_pod( T ) ) && !is_volatile<T>::value > {};
#else
	template < typename T >	struct is_trivially_copyable : bool_constant < ( __has_trivial_copy( T ) || __is_pod( T ) ) && ( !is_volatile<T>::value && !is_reference<T>::value ) > {};
#endif

	template < typename T > struct is_trivial         : bool_constant < __is_trivial( T ) > {};
	template < typename T >	struct is_standard_layout : bool_constant < __is_standard_layout( T ) > {};
	template < typename T > struct is_pod             : bool_constant < __is_pod( T ) > {};
	template < typename T > struct is_literal_type    : bool_constant < __is_literal_type( T ) > {};
	template < typename T > struct is_empty           : bool_constant < __is_empty( T ) > {};
	template < typename T > struct is_polymorphic     : bool_constant < __is_polymorphic( T ) > {};
	template < typename T > struct is_abstract        : bool_constant < __is_abstract( T ) > {};

	template < typename T > struct is_signed : false_type {};
	template <> struct is_signed<char>       : bool_constant< ( char( 0 ) > char( -1 ) ) > {};
	template <> struct is_signed<signed char>            : true_type{};
	template <> struct is_signed<const signed char>      : true_type{};
	template <> struct is_signed<signed short>           : true_type{};
	template <> struct is_signed<const signed short>     : true_type{};
	template <> struct is_signed<signed int>             : true_type{};
	template <> struct is_signed<const signed int>       : true_type{};
	template <> struct is_signed<signed long>            : true_type{};
	template <> struct is_signed<const signed long>      : true_type{};
	template <> struct is_signed<signed long long>       : true_type{};
	template <> struct is_signed<const signed long long> : true_type{};

	template < typename T > struct is_unsigned : false_type {};
	template <> struct is_unsigned<char>       : bool_constant < ( char( 0 ) < char( -1 ) ) > {};
	template <> struct is_unsigned<unsigned char>            : true_type{};
	template <> struct is_unsigned<const unsigned char>      : true_type{};
	template <> struct is_unsigned<unsigned short>           : true_type{};
	template <> struct is_unsigned<const unsigned short>     : true_type{};
	template <> struct is_unsigned<unsigned int>             : true_type{};
	template <> struct is_unsigned<const unsigned int>       : true_type{};
	template <> struct is_unsigned<unsigned long>            : true_type{};
	template <> struct is_unsigned<const unsigned long>      : true_type{};
	template <> struct is_unsigned<unsigned long long>       : true_type{};
	template <> struct is_unsigned<const unsigned long long> : true_type{};

#if NV_COMPILER == NV_MSVC
// 	TODO : implement when needed - this works only in MSVC
// 	template < typename T, typename... Args > 
// 	struct is_constructible : bool_constant < __is_constructible( T, Args ) > {};
// 	template < typename T, typename... Args > 
// 	struct is_trivially_constructible : bool_constant < __is_trivially_constructible( T, Args ) > {};
// 	template < typename T, typename... Args > 
// 	struct is_nothrow_constructible : bool_constant < __is_nothrow_constructible( T, Args ) > {};
#else
// 	template < typename T, typename... Args >
// 	struct is_constructible;
// 	template < typename T, typename... Args >
// 	struct is_trivially_constructible;
// 	template < typename T, typename... Args >
// 	struct is_nothrow_constructible;
#endif

// TODO : implement if needed
// 	template < typename T > 
// 	struct is_default_constructible;
// 	template < typename T > 
// 	struct is_trivially_default_constructible;
// 	template < typename T > 
// 	struct is_nothrow_default_constructible;

// 	namespace detail
// 	{
// 		template < typename T, bool = is_referenceable< T >::value >
// 		struct is_copy_constructible_impl;
// 		template < typename T >
// 		struct is_copy_constructible_impl < T, false > : false_type{};
// 		template < typename T > 
// 		struct is_copy_constructible_impl < T, true > : is_constructible < T, const T& > { };
// 		template < typename T, bool = is_referenceable< T >::value >
// 		struct is_move_constructible_impl;
// 		template < typename T >
// 		struct is_move_constructible_impl < T, false > : false_type{};
// 		template < typename T > 
// 		struct is_move_constructible_impl < T, true > : is_constructible < T, T&& > { };
// 	}

// TODO : implement if needed
// 	template < typename T > 
// 	struct is_copy_constructible;
// 	template < typename T > 
// 	struct is_trivially_copy_constructible;
// 	template < typename T > 
// 	struct is_nothrow_copy_constructible;

// TODO : implement if needed
// 	template < typename T > 
// 	struct is_move_constructible;
// 	template < typename T > 
// 	struct is_trivially_move_constructible;
// 	template < typename T > 
// 	struct is_nothrow_move_constructible;

	namespace detail
	{
		template < typename T >
		struct is_referenceable : bool_constant <
			is_object<T>::value ||
			is_reference<T>::value
		> {};
		template < typename R, typename... Args >
		struct is_referenceable < R( Args... ) > : true_type {};
#if NV_COMPILER != NV_MSVC
		template < typename R, typename... Args >
		struct is_referenceable < R( Args...... ) > : true_type {};
#endif

		template < typename To, typename From >
		struct is_assignable_impl
		{
			struct lazy_int { lazy_int( int ) {} };

			template < typename Dest, typename Source >
			static auto test( int ) -> decltype( ( declval<Dest>() = declval<Source>() ), true_type() );

			template < typename Dest, typename Source >
			static auto test( lazy_int ) -> false_type;

			typedef decltype( test< To, From > ( 0 ) ) type;
		};

		template <
			typename From, typename To, 
			bool Invalid = ( is_void<From>::value || is_function<To>::value || is_array <To>::value )
		>
		struct is_convertible_impl
		{ 
			typedef typename is_void<To>::type type;
		};

		template < typename From, typename To >
		struct is_convertible_impl< From, To, false >
		{
			struct lazy_int { lazy_int( int ) {} };

			template < typename Source >
			static void test_target( Source );

			template < typename Source, typename Dest >
			static auto test( int ) -> decltype( test_target( declval<Source>() ), true_type() );

			template < typename Source, typename Dest >
			static auto test( lazy_int )->false_type;

			typedef decltype( test< From, To >( 0 ) ) type;
		};
	}

	template < typename To, typename From >
	struct is_assignable : detail::is_assignable_impl< To, From >::type { };

	namespace detail
	{
		template < typename T, bool = is_referenceable< T >::value >
		struct is_copy_assignable_impl;
		template < typename T >
		struct is_copy_assignable_impl < T, false > : false_type{};
		template < typename T >
		struct is_copy_assignable_impl < T, true > : is_assignable < T, const T& > {};

		template < typename T, bool = is_referenceable< T >::value >
		struct is_move_assignable_impl;
		template < typename T >
		struct is_move_assignable_impl < T, false > : false_type{};
		template < typename T >
		struct is_move_assignable_impl < T, true > : is_assignable < T&, T&& >{};
	}

	template < typename T >
	struct is_copy_assignable : detail::is_copy_assignable_impl < T > {};

	template < typename T >
	struct is_move_assignable : detail::is_move_assignable_impl < T > {};

// TODO : implement if needed
// 	template < typename T > 
// 	struct is_trivially_assignable;
// 	template < typename T > 
// 	struct is_nothrow_assignable;
// 	template < typename T > 
// 	struct is_trivially_copy_assignable;
// 	template < typename T > 
// 	struct is_nothrow_copy_assignable;
// 	template < typename T > 
// 	struct is_trivially_move_assignable;
// 	template < typename T > 
// 	struct is_nothrow_move_assignable;
// 	template < typename T > 
// 	struct is_trivially_destructible;
// 	template < typename T > 
// 	struct is_nothrow_destructible;

	template < typename T >	struct has_virtual_destructor  : bool_constant < __has_virtual_destructor( T ) > {};

	template< typename T >
	struct alignment_of	: integral_constant < size_t, alignof( typename remove_reference< T >::type ) > {};

	template < typename T >
	struct rank : integral_constant < size_t, 0 > {};
	template < typename T >
	struct rank<T[]> : integral_constant < size_t, rank<T>::value + 1 > {};
	template < typename T, size_t N >
	struct rank<T[N]> : integral_constant < size_t, rank<T>::value + 1 > {};

	template < typename T, unsigned N = 0>
	struct extent : integral_constant < size_t, 0 > {};
	template < typename T>
	struct extent<T[], 0> : integral_constant < size_t, 0 > {};
	template < typename T, unsigned N >
	struct extent<T[], N> : integral_constant < size_t, extent<T, N - 1>::value > {};
	template < typename T, size_t N >
	struct extent<T[N], 0> : integral_constant < size_t, N > {};
	template < typename T, size_t I, unsigned N >
	struct extent< T[I], N > : integral_constant < size_t, extent<T, N - 1>::value > {};

	template < typename Base, typename Derived > 
	struct is_base_of : bool_constant < __is_base_of( Base, Derived ) > {};

#if NV_COMPILER == NV_MSVC 
	template < typename From, typename To > 
	struct is_convertible : bool_constant < __is_convertible_to( From, To ) > {};
#else
	template < typename From, typename To > 
	struct is_convertible : detail::is_convertible_impl< From, To >::type {};
#endif


// TODO: non-standard, remove?
	template < typename T >	struct has_trivial_constructor : bool_constant < __has_trivial_constructor( T ) || __is_pod( T ) > {};
	template < typename T >	struct has_trivial_copy        : bool_constant < ( __has_trivial_copy( T ) || __is_pod( T ) ) && ( !is_volatile<T>::value ) > {};
	template < typename T >	struct has_trivial_assign      : bool_constant < ( __has_trivial_assign( T ) || __is_pod( T ) ) && ( !is_volatile<T>::value && !is_const<T>::value ) > {};
	template < typename T >	struct has_trivial_destructor  : bool_constant < __has_trivial_destructor( T ) || __is_pod( T ) > {};

}

#endif // NV_STL_TRAITS_PROPERTIES_HH
