// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file numeric.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief STL numeric library
*/

#ifndef NV_STL_NUMERIC_HH
#define NV_STL_NUMERIC_HH

#include <nv/core/common.hh>

namespace nv
{

	template < typename InputIterator, typename T >
	inline T accumulate( InputIterator first, InputIterator last, T init )
	{
		while ( first < last )
			init += *first++;
		return init;
	}

	template < typename InputIterator, typename T, typename BinaryFunction >
	inline T accumulate( InputIterator first, InputIterator last, T init, BinaryFunction bin_op )
	{
		while ( first < last )
			init = bin_op( init, *first++ );
		return init;
	}

	template < typename ForwardIterator, typename T >
	inline void iota( ForwardIterator first, ForwardIterator last, T value )
	{
		while ( first < last )
			*first++ = value++;
	}

	template < typename InputIterator1, typename InputIterator2, typename T >
	inline T inner_product( InputIterator1 first1, InputIterator1 last1, InputIterator2 first2, T init )
	{
		while ( first1 < last1 )
			init += *first1++ * *first2++;
		return init;
	}

	template < typename InputIterator1, typename InputIterator2, typename T,
		typename BinaryOperation1, typename BinaryOperation2 >
	inline T inner_product( InputIterator1 first1, InputIterator1 last1, 
		InputIterator2 first2, T init, BinaryOperation1 sum_op, BinaryOperation2 prod_op )
	{
		while ( first1 < last1 )
			init = sum_op( init, prod_op( *first1++, *first2++ ) );
		return init;
	}

	template < typename InputIterator, typename OutputIterator >
	inline OutputIterator partial_sum( InputIterator first, InputIterator last, OutputIterator result )
	{
		if ( first < last )
			*result = *first++;
		while ( first < last )
			*++result = *first++ + *result;
		return result;
	}

	template < typename InputIterator, typename OutputIterator, typename BinaryOperation >
	inline OutputIterator partial_sum( InputIterator first, InputIterator last, OutputIterator result, BinaryOperation sum_op )
	{
		if ( first < last )
			*result = *first++;
		while ( first < last )
			*++result = sum_op( *first++, *result );
		return result;
	}

	template < typename InputIterator, typename OutputIterator >
	inline OutputIterator adjacent_difference( InputIterator first, InputIterator last, OutputIterator result )
	{
		if ( first < last )
			*result++ = *first++;
		while ( first < last )
			*result++ = *first - *( first - 1 );
		return result;
	}

	template < typename InputIterator, typename OutputIterator, typename BinaryOperation >
	inline OutputIterator adjacent_difference( InputIterator first, InputIterator last, OutputIterator result, BinaryOperation diff_op )
	{
		if ( first < last )
			*result++ = *first++;
		while ( first < last )
			*result++ = diff_op( *first, *( first - 1 ) );
		return result;
	}


}

#endif // NV_STL_NUMERIC_HH
