// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file sized_container_handler.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief sized contiguous container handler
*/

#ifndef NV_STL_CONTAINER_SIZED_CONTAINER_HANDLER_HH
#define NV_STL_CONTAINER_SIZED_CONTAINER_HANDLER_HH

#include <nv/core/common.hh>
#include <nv/stl/container/initialize_policy.hh>

namespace nv
{

	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard
	>
	class sized_container_handler : public Storage
	{
	public:
		typedef typename Storage::value_type value_type;
		typedef typename Storage::size_type  size_type;

		inline sized_container_handler() {}
		inline explicit sized_container_handler( size_type new_size ) { resize( new_size ); }
		inline explicit sized_container_handler( default_init ) { resize( default_init() ); }
		inline sized_container_handler( size_type new_size, const value_type& v ) { resize( new_size, v ); }

		// prevent copying 
		sized_container_handler( const sized_container_handler& ) = delete;
		sized_container_handler& operator=( const sized_container_handler& ) = delete;
		// allow move
		sized_container_handler( sized_container_handler&& ) = default;
		sized_container_handler& operator=( sized_container_handler&& ) = default;


		inline void resize( size_type new_size )
		{
			size_type old_size = Storage::size();
			resize_impl( new_size );
			initialize_range( old_size, Storage::size() );
		}

		inline void resize( size_type new_size, default_init )
		{
			size_type old_size = Storage::size();
			resize_impl( new_size );
			initialize_range( old_size, Storage::size(), default_init() );
		}

		inline void resize( size_type new_size, const value_type& value )
		{
			size_type old_size = Storage::size();
			resize_impl( new_size );
			initialize_range( old_size, Storage::size(), value );
		}

		inline void assign( const value_type* ptr, size_type sz )
		{
			if ( Storage::size() > 0 ) InitializePolicy::destroy( Storage::data(), Storage::data() + Storage::size() );
			if ( ptr != nullptr && sz > 0 )
			{
				if ( sz != Storage::size() && Storage::try_resize( sz, false ) )
					InitializePolicy::copy( ptr, ptr + sz, Storage::data() );
			}
			else Storage::try_resize( 0, false );
		}

		template< typename InputIterator >
		inline void assign( InputIterator first, InputIterator last )
		{
			size_type d = distance( first, last );
			if ( d != Storage::size() && Storage::try_resize( sz, false ) )
				InitializePolicy::copy( first, last, Storage::data() );
		}

		// explicit copy
		inline void assign( const sized_container_handler& other )
		{
			assign( other.data(), other.size() );
		}

		inline void clear()
		{
			if ( Storage::size() > 0 )
			{
				InitializePolicy::destroy( Storage::data(), Storage::data() + Storage::size() );
				Storage::try_resize( 0, false );
			}
		}

		~sized_container_handler()
		{
			if ( Storage::size() > 0 ) clear();
		}

	protected:

		inline void initialize_range( size_type old_size, size_type new_size )
		{
			if ( new_size > old_size ) InitializePolicy::initialize( Storage::data() + old_size, Storage::data() + new_size );
		}
		inline void initialize_range( size_type old_size, size_type new_size, default_init )
		{
			if ( new_size > old_size ) uninitialized_construct( Storage::data() + old_size, Storage::data() + new_size );
		}
		inline void initialize_range( size_type old_size, size_type new_size, const value_type& value )
		{
			if ( new_size > old_size ) uninitialized_fill( Storage::data() + old_size, Storage::data() + new_size, value );
		}
		inline void resize_impl( size_type new_size )
		{
			size_type old_size = Storage::size();
			if ( new_size != old_size )
			{
				if ( new_size < old_size )
				{
					InitializePolicy::destroy( Storage::data() + new_size, Storage::data() + old_size );
				}
				if ( Storage::try_resize( new_size, true ) )
				{
					// TODO: error checking
				}
			}
		}

	};

}

#endif // NV_STL_CONTAINER_SIZED_CONTAINER_HANDLER_HH
