// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file reference.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief reference_wrapper class and ref/cref
 */

#ifndef NV_STL_FUNCTIONAL_REFERENCE_HH
#define NV_STL_FUNCTIONAL_REFERENCE_HH

#include <nv/core/common.hh>

namespace nv
{

	template < typename T >
	class reference_wrapper
	{
	public:
		typedef T type;

		reference_wrapper( type& ref ) noexcept : m_pointer( std::addressof( ref ) ) {}
		reference_wrapper( type&& ) = delete;
		reference_wrapper( const reference_wrapper& ) noexcept = default;
		reference_wrapper& operator=( const reference_wrapper& ) noexcept = default;
		operator type& ( ) const noexcept { return *m_pointer; }
		type& get() const noexcept { return *m_pointer; }
	private:
		type* m_pointer;
	};

	template< typename T > 
	inline reference_wrapper< T > ref( T& arg ) noexcept
	{
		return reference_wrapper< T >( arg );
	}

	template< typename T >
	void ref( const T&& ) = delete;

	template< typename T >
	inline reference_wrapper< T > ref( reference_wrapper<T> arg ) noexcept
	{
		return nv::ref( arg.get() );
	}

	template< typename T >
	inline reference_wrapper< const T > cref( const T& arg ) noexcept
	{
		return reference_wrapper< const T >( arg );
	}

	template< typename T >
	void cref( const T&& ) = delete;

	template< typename T >
	inline reference_wrapper< const T > cref( reference_wrapper<T> arg ) noexcept
	{
		return nv::cref( arg.get() );
	}

	template< typename T >
	struct unwrap_reference
	{
		typedef T type;
		static constexpr bool is_wrapped = false;
	};

	template< typename T >
	struct unwrap_reference< reference_wrapper< T > >
	{
		typedef T& type;
		static constexpr bool is_wrapped = true;
	};

}

#endif // NV_STL_FUNCTIONAL_REFERENCE_HH
