// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file fill.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief algorithm fill
*/

#ifndef NV_STL_ALGORITHM_FILL_HH
#define NV_STL_ALGORITHM_FILL_HH

#include <nv/stl/algorithm/common.hh>
#include <nv/stl/algorithm/raw.hh>

namespace nv
{

	namespace detail
	{
		template < typename ForwardIterator, typename T, typename AnyType >
		void fill_impl( ForwardIterator first, ForwardIterator last, const T& value, AnyType, forward_iterator_tag )
		{
			// TODO: version with T temp = value for scalars
			for ( ; first != last; ++first )
				*first = value;
		}

		template < typename BlockAccessIterator, typename T >
		void fill_impl( BlockAccessIterator first, BlockAccessIterator last, T value, true_type, block_access_iterator_tag )
		{
			raw_fill( first, last, (unsigned char)value );
		}

		template < typename ForwardIterator, typename T, typename AnyType >
		ForwardIterator fill_n_impl( ForwardIterator first, size_t count, const T& value, AnyType, forward_iterator_tag )
		{
			// TODO: version with T temp = T() for scalars
			for ( ; count-- > 0; ++first ) *first = value;
			return first;
		}

		template < typename BlockAccessIterator, typename T >
		BlockAccessIterator fill_n_impl( BlockAccessIterator first, size_t count, T value, true_type, block_access_iterator_tag )
		{
			return BlockAccessIterator( raw_fill_n( first, count, (unsigned char)value ) );
		}

		template < typename ForwardIterator, typename T, typename AnyType >
		void fill_default_impl( ForwardIterator first, ForwardIterator last, AnyType, forward_iterator_tag )
		{
			// TODO: version with T temp = T() for scalars
			for ( ; first != last; ++first )
				*first = T();
		}

		template < typename BlockAccessIterator, typename T >
		void fill_default_impl( BlockAccessIterator first, BlockAccessIterator last, true_type, block_access_iterator_tag )
		{
			raw_zero( first, last );
		}

		template < typename ForwardIterator, typename T, typename AnyType >
		ForwardIterator fill_default_n_impl( ForwardIterator first, size_t n, AnyType, forward_iterator_tag )
		{
			// TODO: version with T temp = T() for scalars
			for ( ; n-- > 0; ++first ) *first = value;
			return first;
		}

		template < typename BlockAccessIterator, typename T >
		BlockAccessIterator fill_default_n_impl( BlockAccessIterator first, size_t count, true_type, block_access_iterator_tag )
		{
			return BlockAccessIterator( raw_zero_n( first, count, (unsigned char)value ) );
		}
	}

	template < typename ForwardIterator, typename T >
	inline void fill( ForwardIterator first, ForwardIterator last, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type        value_type;
		typedef typename iterator_traits< ForwardIterator >::iterator_category iterator_category;
		detail::fill_impl( first, last, static_cast<value_type>( value ),
			bool_constant< sizeof( value_type ) == 1 &&
			has_trivial_assign<value_type>::value >(), iterator_category() );
	}

	template < typename ForwardIterator, typename T >
	inline ForwardIterator fill_n( ForwardIterator first, size_t count, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		typedef typename iterator_traits< ForwardIterator >::iterator_category iterator_category;
		return detail::fill_n_impl( first, count, static_cast<value_type>( value ),
			bool_constant< sizeof( value_type ) == 1 && has_trivial_assign<value_type>::value >(), iterator_category() );
	}

	template < typename ForwardIterator >
	inline void fill_default( ForwardIterator first, ForwardIterator last )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		typedef typename iterator_traits< ForwardIterator >::iterator_category iterator_category;
		detail::fill_default_impl( first, last,
			bool_constant< sizeof( value_type ) == 1 && has_trivial_assign<value_type>::value >(), iterator_category() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator fill_default_n( ForwardIterator first, size_t count )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		typedef typename iterator_traits< ForwardIterator >::iterator_category iterator_category;
		return detail::fill_default_n_impl( first, count,
			bool_constant< sizeof( value_type ) == 1 && has_trivial_assign<value_type>::value >(), iterator_category() );
	}

}

#endif // NV_STL_ALGORITHM_FILL_HH
