// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file string_table.hh
 * @author Kornel Kisielewicz
 */

#ifndef NV_IO_STRING_TABLE_HH
#define NV_IO_STRING_TABLE_HH

#include <nv/core/common.hh>
#include <nv/stl/array.hh>
#include <unordered_map>
#include <nv/interface/stream.hh>

namespace nv
{
	class string_table : noncopyable
	{
	public:
		typedef uint16 index;
		typedef uint32 offset;

		string_table( char* data, uint32 size, offset* offsets, index count )
			: m_data( data ), m_size( size ), m_offsets( offsets ), m_count( count )
		{

		}
		explicit string_table( nv::stream* in )
		{
			load( in );
		}

		const char* get( index i ) const
		{
			return i < m_count ? m_data + m_offsets[i] : nullptr;
		}

		void dump( nv::stream* out ) const
		{
			out->write( &m_count,  sizeof( m_count ), 1 );
			out->write( &m_size,   sizeof( m_size ), 1 );
			out->write( m_offsets, sizeof( offset ), m_count );
			out->write( m_data,    sizeof( char ), m_size );
		}

		~string_table()
		{
			delete[] m_data;
			delete[] m_offsets;
		}
	private:
		void load( stream* in )
		{
			in->read( &m_count, sizeof( m_count ), 1 );
			in->read( &m_size, sizeof( m_size ), 1 );
			m_offsets = new offset[ m_count ];
			m_data    = new char[ m_size ];
			in->read( m_offsets, sizeof( offset ), m_count );
			in->read( m_data,    sizeof( char ), m_size );
		}

		char*   m_data;
		uint32  m_size;
		offset* m_offsets;
		index   m_count;
	};

	class string_table_creator
	{
	public:
		typedef string_table::index  index;
		typedef string_table::offset offset;

		string_table_creator();
		index insert( const std::string& s );
		string_table* create_table() const;
		uint32 dump_size() const;
		void dump( nv::stream* out ) const;
		const char* get( index i ) const;
		index get( const std::string& s ) const;
		void clear();
	private:
		std::unordered_map< std::string, index > m_map;
		std::vector< offset > m_offsets;
		std::vector< char >   m_data;
	};


}

#endif // NV_IO_STRING_TABLE_HH
