// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file arithmetic_ops.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief arithmetic function objects
*/

#ifndef NV_STL_FUNCTIONAL_ARITHMETIC_OPS_HH
#define NV_STL_FUNCTIONAL_ARITHMETIC_OPS_HH

#include <nv/stl/functional/common.hh>

namespace nv
{
	template< typename T = void >
	struct plus
	{
		typedef T result_type;
		typedef T first_argument_type;
		typedef T second_argument_type;

		constexpr T operator() ( const T& lhs, const T& rhs ) const
		{
			return lhs + rhs;
		}
	};

	template<>
	struct plus< void >
	{
		typedef is_transparent_t is_transparent;
		
		template < typename T1, typename T2 >
		constexpr auto operator() ( T1&& lhs, T2&& rhs ) const
			-> decltype( static_cast<T1&&>( lhs ) + static_cast<T2&&>( rhs ) )
		{
			return static_cast<T1&&>( lhs ) + static_cast<T2&&>( rhs );
		}
	};

	template< typename T = void >
	struct minus
	{
		typedef T result_type;
		typedef T first_argument_type;
		typedef T second_argument_type;

		constexpr T operator() ( const T& lhs, const T& rhs ) const
		{
			return lhs - rhs;
		}
	};

	template<>
	struct minus< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T1, typename T2 >
		constexpr auto operator() ( T1&& lhs, T2&& rhs ) const
			-> decltype( static_cast<T1&&>( lhs ) - static_cast<T2&&>( rhs ) )
		{
			return static_cast<T1&&>( lhs ) - static_cast<T2&&>( rhs );
		}
	};

	template< typename T = void >
	struct multiplies
	{
		typedef T result_type;
		typedef T first_argument_type;
		typedef T second_argument_type;

		constexpr T operator() ( const T& lhs, const T& rhs ) const
		{
			return lhs * rhs;
		}
	};

	template<>
	struct multiplies< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T1, typename T2 >
		constexpr auto operator() ( T1&& lhs, T2&& rhs ) const
			-> decltype( static_cast<T1&&>( lhs ) * static_cast<T2&&>( rhs ) )
		{
			return static_cast<T1&&>( lhs ) * static_cast<T2&&>( rhs );
		}
	};

	template< typename T = void >
	struct divides
	{
		typedef T result_type;
		typedef T first_argument_type;
		typedef T second_argument_type;

		constexpr T operator() ( const T& lhs, const T& rhs ) const
		{
			return lhs / rhs;
		}
	};

	template<>
	struct divides< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T1, typename T2 >
		constexpr auto operator() ( T1&& lhs, T2&& rhs ) const
			-> decltype( static_cast<T1&&>( lhs ) / static_cast<T2&&>( rhs ) )
		{
			return static_cast<T1&&>( lhs ) / static_cast<T2&&>( rhs );
		}
	};

	template< typename T = void >
	struct modulus
	{
		typedef T result_type;
		typedef T first_argument_type;
		typedef T second_argument_type;

		constexpr T operator() ( const T& lhs, const T& rhs ) const
		{
			return lhs % rhs;
		}
	};

	template<>
	struct modulus< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T1, typename T2 >
		constexpr auto operator() ( T1&& lhs, T2&& rhs ) const
			-> decltype( static_cast<T1&&>( lhs ) % static_cast<T2&&>( rhs ) )
		{
			return static_cast<T1&&>( lhs ) % static_cast<T2&&>( rhs );
		}
	};

	template< typename T = void >
	struct negate
	{
		typedef T result_type;
		typedef T argument_type;

		constexpr T operator() ( const T& arg ) const
		{
			return -arg;
		}
	};

	template<>
	struct negate< void >
	{
		typedef is_transparent_t is_transparent;

		template < typename T >
		constexpr auto operator() ( T&& arg ) const
			-> decltype( -static_cast<T&&>( arg )  )
		{
			return -static_cast<T&&>( arg );
		}
	};

}

#endif // NV_STL_FUNCTIONAL_ARITHMETIC_OPS_HH

