// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file memory.hh
* @author Kornel Kisielewicz
* @brief memory utilities
*/

// TODO: implement const and mutable buffer using the union trick

#ifndef NV_CORE_MEMORY_HH
#define NV_CORE_MEMORY_HH

#include <nv/core/common.hh>
#include <nv/stl/traits/properties.hh>
#include <nv/stl/traits/alignment.hh>
#include <nv/stl/utility.hh>
#include <nv/stl/iterator.hh>
#include <nv/stl/capi.hh>

namespace nv
{

	template< typename T >
	class storage_view
	{
	public:
		typedef T      value_type;
		typedef size_t size_type;
		static const bool is_static = false;
		static const bool is_fixed  = false;
		static const bool is_const  = false;

		NV_CONSTEXPR storage_view() 
			: m_data( nullptr ), m_size( 0 ) {}
		NV_CONSTEXPR storage_view( value_type* a_data, size_type a_size ) 
			: m_data( a_data ), m_size( a_size ) {}

		void assign( value_type* a_data, size_type a_size )
		{
			m_data = a_data;
			m_size = a_size;
		}

		NV_CONSTEXPR size_t size() const { return m_size; }
		NV_CONSTEXPR value_type* data() { return m_data; }
		NV_CONSTEXPR const value_type* data() const { return m_data; }
	protected:
		value_type* m_data;
		size_type   m_size;
	};

	template< typename T >
	class const_storage_view
	{
	public:
		typedef T      value_type;
		typedef size_t size_type;
		static const bool is_static = false;
		static const bool is_fixed  = false;
		static const bool is_const  = true;

		NV_CONSTEXPR const_storage_view() 
			: m_data( nullptr ), m_size( 0 ) {}
		NV_CONSTEXPR const_storage_view( const value_type* a_data, size_type a_size )
			: m_data( a_data ), m_size( a_size ) {}

		void assign( const value_type* a_data, size_type a_size )
		{
			m_data = a_data;
			m_size = a_size;
		}

		NV_CONSTEXPR size_t size() const { return m_size; }
		NV_CONSTEXPR const value_type* data() const { return m_data; }
	protected:
		const value_type* m_data;
		size_type         m_size;
	};


	template< typename T, size_t N >
	class static_storage
	{
	public:
		typedef T value_type;
		static const bool is_static = true;
		static const bool is_fixed  = true;
		static const bool is_const  = false;

		NV_CONSTEXPR T* data() { return static_cast<T*>( m_data ); }
		NV_CONSTEXPR const T* data() const { return static_cast<const T*>( m_data ); }
		static NV_CONSTEXPR size_t capacity() { return N; }
	protected:
		typedef aligned_array_t<T, N, NV_ALIGN_OF( T ) > storage_type;
		storage_type m_data;
	};

	template< typename T, size_t N >
	class fixed_dynamic_storage
	{
	public:
		typedef T value_type;
		static const bool is_static = false;
		static const bool is_fixed = true;
		static const bool is_const = false;

		fixed_dynamic_storage()
		{
			m_data = nvmalloc( N * sizeof( value_type ) );
		}
		~fixed_dynamic_storage()
		{
			nvfree( m_data );
		}
		static NV_CONSTEXPR size_t capacity() { return N; }
		NV_CONSTEXPR T* data() { return static_cast<T*>( m_data ); }
		NV_CONSTEXPR const T* data() const { return static_cast<const T*>( m_data ); }
	protected:
		uint8* m_data;
	};


	namespace detail
	{
		template < typename ForwardIterator, typename T >
		void uninitialized_fill_impl( ForwardIterator first, ForwardIterator last, const T& value, true_type )
		{
			fill( first, last, value );
		}

		template < typename ForwardIterator, typename T >
		void uninitialized_fill_impl( ForwardIterator first, ForwardIterator last, const T& value, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				::new( (void*)&*it ) value_type( value );
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator uninitialized_fill_n_impl( ForwardIterator first, size_t count, const T& value, true_type )
		{
			return fill_n( first, count, value );
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator uninitialized_fill_n_impl( ForwardIterator first, size_t count, const T& value, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				::new ( (void*)&*it ) value_type( value );
			return it;
		}

		template < typename ForwardIterator >
		inline void uninitialized_construct_impl( ForwardIterator first, ForwardIterator last, true_type )
		{
			fill_default( first, last );
		}

		template < typename ForwardIterator >
		inline void uninitialized_construct_impl( ForwardIterator first, ForwardIterator last, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				::new( (void*)&*it ) value_type;
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_construct_n_impl( ForwardIterator first, size_t count, true_type )
		{
			return fill_default_n( first, count );
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_construct_n_impl( ForwardIterator first, size_t count, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				::new( (void*)&*it ) value_type;
		}

		template < typename ForwardIterator >
		inline void uninitialized_destroy_impl( ForwardIterator, ForwardIterator, true_type )
		{
			// no-op
		}

		template < typename ForwardIterator >
		inline void uninitialized_destroy_impl( ForwardIterator first, ForwardIterator last, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				( *it ).~value_type();
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_destroy_n_impl( ForwardIterator first, size_t count, true_type )
		{
			return first + count;
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_destroy_n_impl( ForwardIterator first, size_t count, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				( *it ).~value_type();
			return it;
		}

	}

	template < typename ForwardIterator, typename T >
	inline void uninitialized_fill( ForwardIterator first, ForwardIterator last, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_fill_impl( first, last, value, has_trivial_assign<value_type>() );
	}

	template < typename ForwardIterator, typename T >
	inline void uninitialized_fill_n( ForwardIterator first, size_t count, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_fill_n_impl( first, count, value, has_trivial_assign<value_type>() );
	}

	template < typename T >
	inline void raw_construct_object( void* object )
	{
		new (object)T;
	}

	template < typename T, typename ...Args >
	inline void raw_construct_object( void* object, Args&&... params )
	{
		new (object)T( forward<Args>( params )... );
	}

	template < typename T, typename ...Args >
	inline void construct_object( T* object, Args&&... params )
	{
		new (object)T( forward<Args>( params )... );
	}

	template < typename T >
	inline void destroy_object( T* object )
	{
		object->~T();
	}

	template <typename TYPE>
	void raw_destroy_object( void* object )
	{
		( (TYPE*)object )->TYPE::~TYPE();
	}


	template < typename ForwardIterator, typename ...Args >
	inline void uninitialized_construct( ForwardIterator first, ForwardIterator last, Args&&... params )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		ForwardIterator it( first );
		for ( ; it != last; ++it )
			::new( (void*)&*it ) value_type( forward<Args>( params )... );
	}

	template < typename ForwardIterator >
	inline void uninitialized_construct( ForwardIterator first, ForwardIterator last )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_construct_impl( first, last, has_trivial_constructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator uninitialized_construct_n( ForwardIterator first, size_t count )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_construct_n_impl( first, count, has_trivial_constructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline void uninitialized_destroy( ForwardIterator first, ForwardIterator last )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_destroy_impl( first, last, has_trivial_destructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator uninitialized_destroy_n( ForwardIterator first, size_t count )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_destroy_n_impl( first, count, has_trivial_destructor<value_type>() );
	}

	namespace detail
	{
		template < typename PARENT, typename Storage, bool Const = Storage::is_const >
		class pointer_iterators {};

		template < typename PARENT, typename Storage >
		class pointer_iterators< PARENT, Storage, true >
		{
		public:
			typedef typename Storage::value_type         value_type;
			typedef const value_type*                    const_iterator;
			typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

			inline const_iterator begin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator end() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline const_iterator cbegin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator cend() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
			inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
			inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= static_cast<const PARENT&>( *this ).size(), "Index out of range" ); return begin() + i; }
		};

		template < typename PARENT, typename Storage >
		class pointer_iterators< PARENT, Storage, false >
		{
		public:
			typedef typename Storage::value_type         value_type;
			typedef value_type*                          iterator;
			typedef const value_type*                    const_iterator;
			typedef nv::reverse_iterator<iterator>       reverse_iterator;
			typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

			inline const_iterator begin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator end() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline iterator begin() { return iterator( static_cast<PARENT&>( *this ).data() ); }
			inline iterator end() { return iterator( static_cast<PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline const_iterator cbegin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator cend() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline reverse_iterator rbegin() { return reverse_iterator( end() ); }
			inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
			inline reverse_iterator rend() { return reverse_iterator( begin() ); }
			inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
			inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
			inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= static_cast<const PARENT&>( *this ).size(), "Index out of range" ); return begin() + i; }
			inline iterator       iat( size_t i ) { NV_ASSERT( i <= static_cast<const PARENT&>( *this ).size(), "Index out of range" ); return begin() + i; }
		};

		template < typename Storage, bool Const = Storage::is_const >
		class data_base;


		template < typename Storage >
		class data_base < Storage, true >
			: public pointer_iterators < data_base < Storage, true >, Storage >
		{
		public:
			typedef typename Storage::value_type value_type;
			typedef const value_type*            const_pointer;
			typedef size_t	                     size_type;
			typedef const_pointer                const_iterator;
			typedef const value_type&            const_reference;

			inline data_base() {}
			inline data_base( const_pointer a_data, size_t a_size )
				: m_view( a_data, a_size ) {}
			inline const_pointer data() const { return m_view.data(); }
			inline size_t size() const { return m_view.size(); }
			inline bool empty() const { return size() != 0; }
			inline size_type   raw_size() const { return sizeof( value_type ) * size(); }
			inline const char* raw_data() const { return (const char*)data(); }

			inline const_reference front() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return data()[0]; }
			inline const_reference back() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return data()[size() - 1]; }
		protected:
			void assign( const_pointer a_data, size_type a_size )
			{
				m_view.assign( a_data, a_size );
			}

			Storage m_view;
		};

		template < typename Storage >
		class data_base < Storage, false >
			: public pointer_iterators < data_base < Storage, false >, Storage >
		{
		public:
			typedef typename Storage::value_type value_type;
			typedef value_type*                  pointer;
			typedef const value_type*            const_pointer;
			typedef size_t	                     size_type;
			typedef pointer                      iterator;
			typedef const_pointer                const_iterator;
			typedef value_type&		             reference;
			typedef const value_type&            const_reference;

			inline data_base() {}
			inline data_base( pointer a_data, size_t a_size ) : m_view( a_data, a_size ) {}
			inline const_pointer data() const { return m_view.data(); }
			inline pointer data() { return m_view.data(); }
			inline size_t size() const { return m_view.size(); }
			inline bool empty() const { return size() != 0; }
			inline size_type   raw_size() const { return sizeof( value_type ) * size(); }
			inline const char* raw_data() const { return (const char*)data(); }
			inline char*       raw_data() { return (char*)data(); }

			inline reference       front() { NV_ASSERT( !empty(), "front() called on empty data!" );  return data()[0]; }
			inline const_reference front() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return data()[0]; }
			inline reference       back() { NV_ASSERT( !empty(), "front() called on empty data!" ); return data()[size() - 1]; }
			inline const_reference back() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return data()[size() - 1]; }
		protected:
			void assign( pointer a_data, size_type a_size )
			{
				m_view.assign( a_data, a_size );
			}

			Storage m_view;
		};

	}

	class const_mem_ref : public detail::data_base< const_storage_view< char > >
	{
	public:
		typedef char                 value_type;
		typedef value_type*          pointer;
		typedef const value_type*    const_pointer;
		typedef value_type&		     reference;
		typedef const value_type&    const_reference;
		typedef size_t	             size_type;
		typedef ptrdiff_t            difference_type;
		typedef const_pointer        const_iterator;
	public:
		inline const_mem_ref() : detail::data_base< const_storage_view< char > >() {}
		inline const_mem_ref( const void* p, size_type n ) : detail::data_base< const_storage_view< char > >( (const char*)p, n ) {}
		inline const_mem_ref( const const_mem_ref& l ) : detail::data_base< const_storage_view< char > >( l.data(), l.size() ) {}
	};

	class mem_ref : public detail::data_base< storage_view< char > >
	{
	public:
		typedef char                 value_type;
		typedef value_type*          pointer;
		typedef const value_type*    const_pointer;
		typedef value_type&		     reference;
		typedef const value_type&    const_reference;
		typedef size_t	             size_type;
		typedef ptrdiff_t            difference_type;
		typedef pointer              iterator;
		typedef const_pointer        const_iterator;
	public:
		inline mem_ref() : detail::data_base< storage_view< char > >() {}
		inline mem_ref( void* p, size_type n ) : detail::data_base< storage_view< char > >( (char*)p, n ) {}
		inline mem_ref( const mem_ref& l ) : detail::data_base< storage_view< char > >( const_cast< char* >( l.data() ), l.size() ) {}
	};

	struct policy_initialize_always
	{
		template < typename ForwardIterator >
		static void initialize( ForwardIterator first, ForwardIterator last )
		{
			uninitialized_construct( first, last );
		}
	};

	struct policy_initialize_never
	{
		template < typename ForwardIterator >
		static void initialize( ForwardIterator, ForwardIterator )
		{
		}
	};

	struct policy_initialize_standard
	{
		template < typename ForwardIterator >
		static void initialize( ForwardIterator first, ForwardIterator last )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			if ( !has_trivial_constructor<value_type>() )
				detail::uninitialized_construct_impl( first, last, false_type() );
		}
	};


	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard
	>
	class fixed_container_storage
	{
	public:
		typedef typename Storage::value_type   value_type;

		static const bool   is_static  = Storage::is_static;
		static const bool   is_fixed   = Storage::is_fixed;
		static const bool   is_const   = Storage::is_const;
		static const size_t type_size  = sizeof( value_type );

		fixed_container_storage()
		{
			InitializePolicy::initialize( data(), data() + m_data.capacity() );
		}

		~fixed_container_storage()
		{
			uninitialized_destroy( data(), data() + m_data.capacity() );
		}

		static NV_CONSTEXPR size_t capacity() { return Storage::capacity(); }
		static NV_CONSTEXPR size_t size() { return Storage::capacity(); }
		NV_CONSTEXPR value_type* data() { return m_data.data(); }
		NV_CONSTEXPR const value_type* data() const { return m_data.data(); }
	protected:
		Storage m_data;
	};

	template < typename T >
	class dynamic_container_allocator
	{
	public:
		static const bool   is_static = false;
		static const bool   is_fixed  = true;
		static const size_t type_size = sizeof( T );

		dynamic_container_allocator()
			: m_data(nullptr), m_size(0)
		{
		
		}

		static NV_CONSTEXPR size_t capacity() { return 0x80000000; }
		size_t size() const { return m_size; }
		T* data() { return static_cast<T*>( m_data ); }
		const T* data() const { return static_cast<const T*>( m_data ); }

		~dynamic_container_allocator()
		{
			delete m_data;
		}

		uint8* m_data;
		size_t m_size;
	};

}

#endif // NV_CORE_MEMORY_HH
