// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file limits.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief STL limits library
*/

#ifndef NV_STL_LIMITS_HH
#define NV_STL_LIMITS_HH

#include <nv/core/common.hh>

namespace nv
{
	struct limits
	{
		static const unsigned char uc_min = 0;
		static const unsigned char uc_max = 255;
		static const unsigned char uc_bit = sizeof( unsigned char ) * 8;
		static const signed char sc_min = -128;
		static const signed char sc_max = 127;
		static const signed char sc_bit = sizeof( signed char ) * 8;
		static const char c_min = ( char( 0 ) < char( -1 ) ? 0 : -128 );
		static const char c_max = ( char( 0 ) < char( -1 ) ? 255 : 127 );
		static const char c_bit = sizeof( char ) * 8;
		static const unsigned short us_min = 0;
		static const unsigned short us_max = 65535;
		static const unsigned short us_bit = sizeof( unsigned short ) * 8;
		static const signed short ss_min = -32768;
		static const signed short ss_max = 32767;
		static const signed short ss_bit = sizeof( signed short ) * 8;
		static const unsigned int ui_min = 0;
		static const unsigned int ui_max = 4294967295;
		static const unsigned int ui_bit = sizeof( unsigned int ) * 8;
		static const signed int si_min = -2147483648;
		static const signed int si_max = 2147483647;
		static const signed int si_bit = sizeof( signed int ) * 8;
		static const unsigned long ul_min = 0;
		static const unsigned long ul_max = 4294967295;
		static const unsigned long ul_bit = sizeof( unsigned long ) * 8;
		static const signed long sl_min = -2147483648;
		static const signed long sl_max = 2147483647;
		static const signed long sl_bit = sizeof( signed long ) * 8;
		static const unsigned long long ull_min = 0;
		static const unsigned long long ull_max = 18446744073709551615;
		static const unsigned long long ull_bit = sizeof( unsigned long long ) * 8;
		static const signed long long sll_min = -9223372036854775808;
		static const signed long long sll_max = 9223372036854775807;
		static const signed long long sll_bit = sizeof( signed long long ) * 8;

		static const int f_dig        = 6;
		static const int f_mant_dig   = 24;
		static const int f_max_exp    = 128;
		static const int f_max_10_exp = 38;
		static const int f_min_exp    = -125;
		static const int f_min_10_exp = -37;

		static const int d_dig        = 15;
		static const int d_mant_dig   = 53;
		static const int d_max_exp    = 1024;
		static const int d_max_10_exp = 308;
		static const int d_min_exp    = -1021;
		static const int d_min_10_exp = -307;

		static const int ld_dig        = d_dig;
		static const int ld_mant_dig   = d_mant_dig;
		static const int ld_max_exp    = d_max_exp;
		static const int ld_max_10_exp = d_max_10_exp;
		static const int ld_min_exp    = d_min_exp;
		static const int ld_min_10_exp = d_min_10_exp;
	};

	enum float_denorm_style
	{
		denorm_indeterminate = -1,
		denorm_absent        = 0,
		denorm_present       = 1
	};

	enum float_round_style
	{
		round_indeterminate       = -1,
		round_toward_zero         = 0,
		round_to_nearest          = 1,
		round_toward_infinity     = 2,
		round_toward_neg_infinity = 3
	};

	namespace detail
	{
		struct numeric_limits_base
		{
			static const float_denorm_style has_denorm = denorm_absent;
			static const bool has_denorm_loss   = false;
			static const bool has_infinity      = false;
			static const bool has_quiet_NaN     = false;
			static const bool has_signaling_NaN = false;
			static const bool is_bounded        = false;
			static const bool is_exact          = false;
			static const bool is_iec559         = false;
			static const bool is_integer        = false;
			static const bool is_modulo         = false;
			static const bool is_signed         = false;
			static const bool is_specialized    = false;
			static const bool tinyness_before   = false;
			static const bool traps             = false;
			static const float_round_style round_style = round_toward_zero;
			static const int digits         = 0;
			static const int digits10       = 0;
			static const int max_digits10   = 0;
			static const int max_exponent   = 0;
			static const int max_exponent10 = 0;
			static const int min_exponent   = 0;
			static const int min_exponent10 = 0;
			static const int radix          = 0;
		};

		struct numeric_limits_int_base : public numeric_limits_base
		{
			static const bool is_bounded     = true;
			static const bool is_exact       = true;
			static const bool is_integer     = true;
			static const bool is_modulo      = true;
			static const bool is_specialized = true;
			static const int radix = 2;
		};

		struct numeric_limits_float_base : public numeric_limits_base
		{
			static const float_denorm_style has_denorm = denorm_present;
			static const bool has_denorm_loss   = true;
			static const bool has_infinity      = true;
			static const bool has_quiet_NaN     = true;
			static const bool has_signaling_NaN = true;
			static const bool is_bounded        = true;
			static const bool is_iec559         = true;
			static const bool is_signed         = true;
			static const bool is_specialized    = true;
			static const bool tinyness_before   = true;
			static const float_round_style round_style = round_to_nearest;
			static const int radix = 2;
		};

	}

	template< typename T >
	struct numeric_limits : detail::numeric_limits_base
	{
		static NV_CONSTEXPR T min()           NV_NOEXCEPT { return T( 0 ); }
		static NV_CONSTEXPR T max()           NV_NOEXCEPT { return T( 0 ); }
		static NV_CONSTEXPR T lowest()        NV_NOEXCEPT { return T( 0 ); }
		static NV_CONSTEXPR T epsilon()       NV_NOEXCEPT { return T( 0 ); }
		static NV_CONSTEXPR T round_error()   NV_NOEXCEPT { return T( 0 ); }
	};

	template< typename T >
	struct numeric_limits< const T > : numeric_limits < T > {};

	template< typename T >
	struct numeric_limits< volatile T > : numeric_limits < T > {};

	template< typename T >
	struct numeric_limits< const volatile T > : numeric_limits < T > {};

	template<>
	struct numeric_limits < bool > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR bool min()           NV_NOEXCEPT { return false; }
		static NV_CONSTEXPR bool max()           NV_NOEXCEPT { return true; }
		static NV_CONSTEXPR bool lowest()        NV_NOEXCEPT { return false; }
		static NV_CONSTEXPR bool epsilon()       NV_NOEXCEPT { return bool( 0 ); }
		static NV_CONSTEXPR bool round_error()   NV_NOEXCEPT { return bool( 0 ); }

		static const bool is_modulo = false;
		static const bool is_signed = false;
		static const int  digits    = 1;
		static const int  digits10  = 0;
	};

	template<>
	struct numeric_limits < char > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR char min()           NV_NOEXCEPT { return limits::c_min; }
		static NV_CONSTEXPR char max()           NV_NOEXCEPT { return limits::c_max; }
		static NV_CONSTEXPR char lowest()        NV_NOEXCEPT { return limits::c_min; }
		static NV_CONSTEXPR char epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR char round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = (limits::c_min < 0);
		static const int  digits    = limits::c_bit - ( limits::c_min < 0 ? 1 : 0 );
		static const int  digits10  = (limits::c_bit - ( limits::c_min < 0 ? 1 : 0 )) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned char > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR unsigned char min()           NV_NOEXCEPT { return limits::uc_min; }
		static NV_CONSTEXPR unsigned char max()           NV_NOEXCEPT { return limits::uc_max; }
		static NV_CONSTEXPR unsigned char lowest()        NV_NOEXCEPT { return limits::uc_min; }
		static NV_CONSTEXPR unsigned char epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR unsigned char round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = false;
		static const int  digits    = limits::c_bit;
		static const int  digits10  = limits::c_bit  * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed char > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR signed char min()           NV_NOEXCEPT { return limits::sc_min; }
		static NV_CONSTEXPR signed char max()           NV_NOEXCEPT { return limits::sc_max; }
		static NV_CONSTEXPR signed char lowest()        NV_NOEXCEPT { return limits::sc_min; }
		static NV_CONSTEXPR signed char epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR signed char round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = true;
		static const int  digits    = limits::c_bit - 1;
		static const int  digits10  = (limits::c_bit - 1) * 301 / 1000;
	};


	template<>
	struct numeric_limits < unsigned short > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR unsigned short min()           NV_NOEXCEPT { return limits::us_min; }
		static NV_CONSTEXPR unsigned short max()           NV_NOEXCEPT { return limits::us_max; }
		static NV_CONSTEXPR unsigned short lowest()        NV_NOEXCEPT { return limits::us_min; }
		static NV_CONSTEXPR unsigned short epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR unsigned short round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = false;
		static const int  digits    = limits::us_bit;
		static const int  digits10  = limits::us_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed short > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR signed short min()           NV_NOEXCEPT { return limits::ss_min; }
		static NV_CONSTEXPR signed short max()           NV_NOEXCEPT { return limits::ss_max; }
		static NV_CONSTEXPR signed short lowest()        NV_NOEXCEPT { return limits::ss_min; }
		static NV_CONSTEXPR signed short epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR signed short round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = true;
		static const int  digits    = limits::ss_bit - 1;
		static const int  digits10  = ( limits::ss_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned int > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR unsigned int min()           NV_NOEXCEPT { return limits::ui_min; }
		static NV_CONSTEXPR unsigned int max()           NV_NOEXCEPT { return limits::ui_max; }
		static NV_CONSTEXPR unsigned int lowest()        NV_NOEXCEPT { return limits::ui_min; }
		static NV_CONSTEXPR unsigned int epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR unsigned int round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = false;
		static const int  digits    = limits::ui_bit;
		static const int  digits10  = limits::ui_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed int > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR signed int min()           NV_NOEXCEPT { return limits::si_min; }
		static NV_CONSTEXPR signed int max()           NV_NOEXCEPT { return limits::si_max; }
		static NV_CONSTEXPR signed int lowest()        NV_NOEXCEPT { return limits::si_min; }
		static NV_CONSTEXPR signed int epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR signed int round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = true;
		static const int  digits    = limits::si_bit - 1;
		static const int  digits10  = ( limits::si_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned long > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR unsigned long min()           NV_NOEXCEPT { return limits::ul_min; }
		static NV_CONSTEXPR unsigned long max()           NV_NOEXCEPT { return limits::ul_max; }
		static NV_CONSTEXPR unsigned long lowest()        NV_NOEXCEPT { return limits::ul_min; }
		static NV_CONSTEXPR unsigned long epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR unsigned long round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = false;
		static const int  digits    = limits::ul_bit;
		static const int  digits10  = limits::ul_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed long > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR signed long min()           NV_NOEXCEPT { return limits::sl_min; }
		static NV_CONSTEXPR signed long max()           NV_NOEXCEPT { return limits::sl_max; }
		static NV_CONSTEXPR signed long lowest()        NV_NOEXCEPT { return limits::sl_min; }
		static NV_CONSTEXPR signed long epsilon()       NV_NOEXCEPT { return 0; }
		static NV_CONSTEXPR signed long round_error()   NV_NOEXCEPT { return 0; }

		static const bool is_signed = true;
		static const int  digits    = limits::sl_bit - 1;
		static const int  digits10  = ( limits::sl_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < unsigned long long > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR unsigned long long min()           NV_NOEXCEPT { return limits::ull_min; }
		static NV_CONSTEXPR unsigned long long max()           NV_NOEXCEPT{ return limits::ull_max; }
		static NV_CONSTEXPR unsigned long long lowest()        NV_NOEXCEPT{ return limits::ull_min; }
		static NV_CONSTEXPR unsigned long long epsilon()       NV_NOEXCEPT{ return 0; }
		static NV_CONSTEXPR unsigned long long round_error()   NV_NOEXCEPT{ return 0; }

		static const bool is_signed = false;
		static const int  digits    = limits::ull_bit;
		static const int  digits10  = limits::ull_bit * 301 / 1000;
	};

	template<>
	struct numeric_limits < signed long long > : detail::numeric_limits_int_base
	{
		static NV_CONSTEXPR signed long long min()           NV_NOEXCEPT{ return limits::sll_min; }
		static NV_CONSTEXPR signed long long max()           NV_NOEXCEPT{ return limits::sll_max; }
		static NV_CONSTEXPR signed long long lowest()        NV_NOEXCEPT{ return limits::sll_min; }
		static NV_CONSTEXPR signed long long epsilon()       NV_NOEXCEPT{ return 0; }
		static NV_CONSTEXPR signed long long round_error()   NV_NOEXCEPT{ return 0; }

		static const bool is_signed = true;
		static const int  digits    = limits::sll_bit - 1;
		static const int  digits10  = ( limits::sll_bit - 1 ) * 301 / 1000;
	};

	template<>
	struct numeric_limits < float > : detail::numeric_limits_float_base
	{
		static NV_CONSTEXPR float min()           NV_NOEXCEPT { return 1.175494351e-38F; }
		static NV_CONSTEXPR float max()           NV_NOEXCEPT { return 3.402823466e+38F; }
		static NV_CONSTEXPR float lowest()        NV_NOEXCEPT { return -max(); }
		static NV_CONSTEXPR float epsilon()       NV_NOEXCEPT { return 1.192092896e-07F; }
		static NV_CONSTEXPR float round_error()   NV_NOEXCEPT { return 0.5F; }

		static const int digits         = limits::f_mant_dig;
		static const int digits10       = limits::f_dig;
		static const int max_digits10   = 2 + limits::f_dig * 301 / 1000;
		static const int max_exponent   = limits::f_max_exp;
		static const int max_exponent10 = limits::f_max_10_exp;
		static const int min_exponent   = limits::f_min_exp;
		static const int min_exponent10 = limits::f_min_10_exp;
	};

	template<>
	struct numeric_limits < double > : detail::numeric_limits_float_base
	{
		static NV_CONSTEXPR double min()           NV_NOEXCEPT { return 2.2250738585072014e-308; }
		static NV_CONSTEXPR double max()           NV_NOEXCEPT { return 1.7976931348623158e+308; }
		static NV_CONSTEXPR double lowest()        NV_NOEXCEPT { return -max(); }
		static NV_CONSTEXPR double epsilon()       NV_NOEXCEPT { return 2.2204460492503131e-016; }
		static NV_CONSTEXPR double round_error()   NV_NOEXCEPT { return 0.5; }

		static const int digits         = limits::d_mant_dig;
		static const int digits10       = limits::d_dig;
		static const int max_digits10   = 2 + limits::d_dig * 301 / 1000;
		static const int max_exponent   = limits::d_max_exp;
		static const int max_exponent10 = limits::d_max_10_exp;
		static const int min_exponent   = limits::d_min_exp;
		static const int min_exponent10 = limits::d_min_10_exp;
	};

	template<>
	struct numeric_limits < long double > : detail::numeric_limits_float_base
	{
		static NV_CONSTEXPR long double min()           NV_NOEXCEPT { return 2.2250738585072014e-308; }
		static NV_CONSTEXPR long double max()           NV_NOEXCEPT { return 1.7976931348623158e+308; }
		static NV_CONSTEXPR long double lowest()        NV_NOEXCEPT { return -max(); }
		static NV_CONSTEXPR long double epsilon()       NV_NOEXCEPT { return 2.2204460492503131e-016; }
		static NV_CONSTEXPR long double round_error()   NV_NOEXCEPT { return 0.5; }

		static const int digits         = limits::ld_mant_dig;
		static const int digits10       = limits::ld_dig;
		static const int max_digits10   = 2 + limits::ld_dig * 301 / 1000;
		static const int max_exponent   = limits::ld_max_exp;
		static const int max_exponent10 = limits::ld_max_10_exp;
		static const int min_exponent   = limits::ld_min_exp;
		static const int min_exponent10 = limits::ld_min_10_exp;
	};

}

#endif // NV_STL_LIMITS_HH