// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file capi.hh
* @author Kornel Kisielewicz
* @brief C API functions
*
* This header is temporary
*/

#ifndef NV_STL_CAPI_HH
#define NV_STL_CAPI_HH

#include <nv/core/common.hh>

namespace nv
{

	namespace detail
	{

#if NV_COMPILER == NV_MSVC
#define NV_CAPI_CALL(name) detail::name
		extern "C" {
			NV_NOALIAS NV_RESTRICT void * __cdecl calloc( size_t _Count, size_t _Size );
			NV_NOALIAS             void   __cdecl free( void * _Memory );
			NV_NOALIAS NV_RESTRICT void * __cdecl malloc( size_t );
			NV_NOALIAS NV_RESTRICT void * __cdecl realloc( void * _Memory, size_t _NewSize );
			void *  __cdecl memcpy( void * _Dst, const void * _Src, size_t _Size );
			void *  __cdecl memmove( void * _Dst, const void * _Src, size_t _Size );
			void *  __cdecl memset( void * _Dst, int _Val, size_t _Size );
			void *  __cdecl memchr( const void * _Buf, int _Val, size_t _MaxCount );
			int     __cdecl memcmp( const void * _Buf1, const void * _Buf2, size_t _Size );
			int     __cdecl strcmp( const char * _Str1, const char * _Str2 );
			int     __cdecl strncmp( const char * _Str1, const char * _Str2, size_t _MaxCount );
			size_t  __cdecl strlen( const char * _Str );
		}
#endif 
		// TODO: remove when clang 3.5 will be used (with builtins)
#if NV_COMPILER == NV_CLANG
#define NV_CAPI_CALL(name) detail::name
		extern "C" {
			extern void * __cdecl calloc( nv::size_t, nv::size_t );
			extern void   __cdecl free( void * );
			extern void * __cdecl malloc( nv::size_t );
			extern void * __cdecl realloc( void * , nv::size_t );
			extern void * __cdecl memcpy( void * , const void * , nv::size_t );
			extern void * __cdecl memmove( void * , const void * , nv::size_t );
			extern void * __cdecl memset( void * , int , nv::size_t );
			extern void * __cdecl memchr( const void * , int , nv::size_t );
			extern int    __cdecl memcmp( const void * , const void * , nv::size_t  );
			extern int    __cdecl strcmp ( const char * , const char * );
			extern int    __cdecl strncmp( const char * , const char * , size_t );
			extern nv::size_t __cdecl strlen ( const char * );
	}
#endif

#if NV_COMPILER == NV_GNUC
#define NV_CAPI_CALL(name) __builtin_##name
#endif
	}

	inline void* nvmalloc( size_t size )
	{
		return NV_CAPI_CALL(malloc)( size );
	}
	inline void nvfree( void* p )
	{
		NV_CAPI_CALL( free )( p );
	}

	inline void* nvmemcpy( void *dest, const void *src, size_t count )
	{
		return NV_CAPI_CALL( memcpy )( dest, src, count );
	}

	inline void* nvmemmove( void *dest, const void *src, size_t count )
	{
		return NV_CAPI_CALL( memmove )( dest, src, count );
	}

	inline void* nvmemset( void *dest, unsigned char value, size_t count )
	{
		return NV_CAPI_CALL( memset )( dest, (int)value, count );
	}

	inline void* nvmemchr( const void *src, unsigned char value, size_t max_count )
	{
		return NV_CAPI_CALL( memchr )( src, (int)value, max_count );
	}

	inline int nvmemcmp( const void * m1, const void * m2, nv::size_t max_count )
	{
		return NV_CAPI_CALL( memcmp )( m1, m2, max_count );
	}

	inline int nvstrcmp( const char * s1, const char * s2 )
	{
		return NV_CAPI_CALL( strcmp )( s1, s2 );
	}	

	inline int nvstrncmp( const char * s1, const char * s2, size_t max_count )
	{
		return NV_CAPI_CALL( strncmp )( s1, s2, max_count );
	}

	inline nv::size_t nvstrlen( const char * s )
	{
		return NV_CAPI_CALL( strlen )( s );
	}

}

#undef NV_CAPI_CALL

#endif // NV_STL_CAPI_HH
