// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_IMAGE_HH
#define NV_IMAGE_HH

#include <nv/core/common.hh>
#include <nv/stl/math.hh>
#include <nv/interface/image_data.hh>

namespace nv
{
	/**
	 * Defines the starting position of some region and its size.
	 */
	struct region
	{
		ivec2 pos;
		ivec2 size;

		region() : pos(0,0), size(0,0) {}
		region( ivec2 apos, ivec2 asize ) : pos(apos), size(asize) {}
	};

	/**
	 * Holder for images.
	 *
	 * By an image we understand a group of pixels, limited to X and Y
	 * dimensions, of a certain depth (by depth we mean how rich
	 * the colors are).
	 */
	class image
	{
	public:
		/**
		 * Constructor
		 *
		 * @arg[in] size : Dimensions of the image as a vector.
		 * @arg[in] depth :  Depth of the image.
		 */
		image( ivec2 size, size_t depth );
		/**
		 * Constructor
		 *
		 * @arg[in] data : image data to be used - will be released, but not deleted!
		 */
		explicit image( image_data* data );
		/**
		 * Full constructor.
		 *
		 * Allows creation of an image with specified data. It is
		 * to reverse the image (from left to right).
		 *
		 * @arg[in] size : Dimensions of the image as a vector.
		 * @arg[in] depth : Depth of the image.
		 * @arg[in] data : Data of the image.
		 * @arg[in] reversed : Determines if the image should be
		 * 	reversed or not.
		 */
		image( ivec2 size, size_t depth, const uint8 * data, bool reversed = false );
		/**
		 * Fills the image with a given value.
		 *
		 * @arg[in] value : Value to fill.
		 */
		void fill( uint8 value );
		/**
		 * Fills the region with a given value.
		 *
		 * @arg[in] r : Region to be filled.
		 * @arg[in] value : Value to fill.
		 */
		void fill( region r, uint8 value, int stride = 0 );
		/**
		 * Just like fill, but a region can be specified.
		 *
		 * @arg[in] r : Region to be filled.
		 * @arg[in] data : Data to fill the region
		 * @arg[in] stride : 
		 */
		void set_region( region r, const uint8 * data, int stride = 0 );

		/**
		 * Just like fill, but a region can be specified. Checks for data depth,
		 * and converts if needed.
		 *
		 * @arg[in] r : Region to be filled.
		 * @arg[in] data : Data to fill the region
		 */
		void set_region( region r, const image_data* idata );

		/**
		 * Default destructor. Deallocates data.
		 */
		virtual ~image();
		/**
		 * Getter for data.
		 *
		 * @return The data.
		 */
		const uint8 * get_data() const { return m_data; }
		/**
		 * Getter for size.
		 *
		 * @return Returns the size of the image as a vector.
		 */
		ivec2 get_size() const { return m_size; }
		/**
		 * Getter for depth.
		 *
		 * @return Returns depth of the image.
		 */
		size_t get_depth() const { return m_depth; }
	protected:
		ivec2  m_size;  //!< Defines the size of the image as a vector.
		size_t m_depth; //!< Defines the depth of the image
		uint8* m_data;  //!< Holder for data
	};

}

#endif // NV_IMAGE_HH

