// Copyright (C) 2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file resource_manager.hh
 * @author Kornel Kisielewicz
 * @brief engine resource manager
 */

#ifndef NV_ENGINE_RESOURCE_MANAGER_HH
#define NV_ENGINE_RESOURCE_MANAGER_HH

#include <nv/core/common.hh>
#include <nv/interface/context.hh>
#include <nv/lua/lua_state.hh>
#include <unordered_map>

namespace nv
{

	typedef uint32 resource_id;
	typedef uint32 resource_type_id;

	class resource_system;

	class resource_manager_base
	{
	public:	
		resource_manager_base() : m_lua( nullptr ) {}
		void initialize( lua::state* state );
		virtual string_ref get_storage_name() const = 0;
		virtual string_ref get_resource_name() const = 0;
		virtual void clear() { m_names.clear(); }
		void load_all();
		resource_id load_resource( const std::string& id );
		virtual ~resource_manager_base() {}
	protected:
		virtual resource_id load_resource( lua::table_guard& table ) = 0;

		lua::state* m_lua;
		std::unordered_map< string, resource_id > m_names;
	};

	template < typename T >
	class resource_manager : public resource_manager_base
	{
	public:
		resource_manager() { m_data.push_back(T()); }
		T get_resource( resource_id id )
		{
			return ( id < m_data.size() ? m_data[ id ] : T() );
		}
		T get_resource( const string& id )
		{
			auto m = m_names.find( id );
			if ( m != m_names.end() )
			{
				return get_resource( m->second );
			}
			return get_resource( load_resource( id ) );
		}
		virtual void clear()
		{
			resource_manager_base::clear();
			for ( uint32 i = 1; i < m_data.size(); ++i )
				release( m_data[i] );
			m_data.clear();
			m_data.push_back( T() );
		}
		virtual ~resource_manager() 
		{
			clear();
		}
	protected:
		virtual void release( T ) {}

		resource_id add( T resource )
		{
			m_data.push_back( resource );
			return m_data.size() - 1;
		}

		std::vector< T > m_data;
	};


	class resource_system
	{
	public:
		explicit resource_system() : m_lua_state( nullptr ) { m_managers.push_back(nullptr); }
		resource_type_id register_resource_type( const string& name, resource_manager_base* manager );
		resource_type_id get_resource_type_id( const string& name ) const;
		void initialize( lua::state* a_lua_state );
		virtual ~resource_system();
	protected:
		std::vector< resource_manager_base* >     m_managers;
		std::unordered_map< string, resource_id > m_manager_names;
		lua::state* m_lua_state; 
	};

}

#endif // NV_ENGINE_RESOURCE_MANAGER_HH
