// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_CORE_COMMON_HH
#define NV_CORE_COMMON_HH

// NV Library version
#define NV_VERSION_MAJOR 0
#define NV_VERSION_MINOR 1
#define NV_VERSION_PATCH 0
#define NV_VERSION    (NV_VERSION_MAJOR << 16) | (NV_VERSION_MINOR << 8) | NV_VERSION_PATCH)

// Platform
#define NV_WINDOWS        1
#define NV_LINUX          2
#define NV_APPLE          3

// Compiler
#define NV_MSVC           1
#define NV_GNUC           2
#define NV_CLANG          3

// Endianess
#define NV_LITTLEENDIAN   1
#define NV_BIGENDIAN      2

// Bits
#define NV_32BIT          1
#define NV_64BIT          2

// Assumption
#define NV_ENDIANESS NV_LITTLEENDIAN

// Platform detection
#if defined( __WIN32__ ) || defined( _WIN32 )
#define NV_PLATFORM NV_WINDOWS
#elif defined( __APPLE_CC__)
#define NV_PLATFORM NV_APPLE
#else
#define NV_PLATFORM NV_LINUX
#endif

// Compiler detection
#if defined( _MSC_VER )
#define NV_COMPILER NV_MSVC
#define NV_COMP_VER _MSC_VER
#elif defined( __clang__ )
#define NV_COMPILER NV_CLANG
#define NV_COMP_VER (((__clang_major__)*100) + (__clang_minor__*10) + __clang_patchlevel__)
#elif defined( __GNUC__ )
#define NV_COMPILER NV_GNUC
#define NV_COMP_VER (((__GNUC__)*100) + (__GNUC_MINOR__*10) + __GNUC_PATCHLEVEL__)
#else
#error "Unknown compiler!"
#endif

// Architecture detection
#if defined(__x86_64__) || defined(_M_X64) || defined(__powerpc64__) || defined(__alpha__)
#define NV_ARCHITECTURE NV_64BIT
#elif defined(__ia64__) || defined(__s390__) || defined(__s390x__)
#define NV_ARCHITECTURE NV_64BIT
#else
#define NV_ARCHITECTURE NV_32BIT
#endif

// Platform specific settings.
#if NV_PLATFORM == NV_WINDOWS
#ifdef _DEBUG
#define NV_DEBUG 1
#else
#define NV_DEBUG 0
#endif
#endif

#if NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#ifdef DEBUG
#define NV_DEBUG 1
#else
#define NV_DEBUG 0
#endif
#endif

#if NV_DEBUG
#ifndef NV_PROFILER
#define NV_PROFILER 
#endif
#endif

#ifdef NV_PROFILER
#undef NV_PROFILER
#define NV_PROFILER 1
#else
#define NV_PROFILER 0
#endif

#if NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#define NV_POSIX
#endif

#if NV_COMPILER == NV_MSVC && NV_COMP_VER < 1700
#error "MSVC 2013+ required!"
#endif

#if NV_COMPILER == NV_GNUC && NV_COMP_VER < 460
#error "GCC 4.6+ required!"
#endif

#if NV_COMPILER == NV_CLANG && NV_COMP_VER < 320
#error "clang 3.2+ required!"
#endif

// Feature incoming - remove and find when constexpr compiler is used
#define NV_CONSTEXPR 
#define NV_CONSTEXPR_CONST const

#if NV_COMPILER == NV_MSVC 
#pragma warning( disable : 4201 ) // nonstandard extension used : nameless struct/union
#pragma warning( disable : 4510 ) // default constructor could not be generated
#pragma warning( disable : 4610 ) // object 'class' can never be instantiated - user-defined constructor required
#undef _SCL_SECURE_NO_WARNINGS // to prevent redefining
#define _SCL_SECURE_NO_WARNINGS
#endif

#include <cassert>

#define NV_STRINGIZE_DETAIL(x) #x
#define NV_STRINGIZE(x) NV_STRINGIZE_DETAIL(x)

#if NV_COMPILER == NV_MSVC
#define NV_DEPRECATED(func) __declspec(deprecated) func
#define NV_ALIGN(value) __declspec(align(value))
#define NV_ALIGNED_STRUCT(value) __declspec(align(value)) struct
#define NV_RESTRICT __declspec(restrict)
#define NV_RESTRICT_VAR __restrict
#define NV_NOEXCEPT throw()
//#define NV_CONSTEXPR 
#elif NV_COMPILER == NV_GNUC || NV_COMPILER == NV_CLANG
#define NV_DEPRECATED(func) func __attribute__ ((deprecated))
#define NV_ALIGN(value) __attribute__((aligned(value)))
#define NV_ALIGNED_STRUCT(value) struct __attribute__((aligned(value)))
#define NV_RESTRICT __restrict__
#define NV_RESTRICT_VAR __restrict__
#define NV_NOEXCEPT noexcept
//#define NV_CONSTEXPR constexpr 
#else
#define NV_DEPRECATED(func) func
#define NV_ALIGN(value)
#define NV_ALIGNED_STRUCT(value) struct 
#define NV_RESTRICT
#define NV_RESTRICT_VAR
#define NV_NOEXCEPT 
//#define NV_CONSTEXPR 
#endif 

#define NV_UNUSED(x) (void)(x)
#define NV_ASSERT(cond, msg) assert( (cond) && msg )
#define NV_THROW(eobj, ...) { \
	NV_LOG_ERROR( __FILE__ " line " NV_STRINGIZE(__LINE__) " - exception thrown - " #eobj ); \
	throw eobj( __VA_ARGS__ ); \
} 

// MSVC and GCC is too stupid to notice fully covered enums, clang 
// is picky about it
#if NV_COMPILER == NV_CLANG
#define NV_RETURN_COVERED_DEFAULT( value ) 
#else
#define NV_RETURN_COVERED_DEFAULT( value ) default : return value
#endif

#define NV_COUNT_OF(x) ((sizeof(x)/sizeof(0[x])) / ((size_t)(!(sizeof(x) % sizeof(0[x])))))
#define NV_SAFE_ARRAY( arr, idx, def ) ( index < NV_COUNT_OF(arr) ? (arr)[idx] : (def) )

namespace nv
{
	namespace lua
	{
		class state;
	}

	// Typedefs for fixed size types.
	typedef signed   char      sint8;
	typedef signed   short     sint16;
	typedef signed   int       sint32;

	typedef unsigned char      uint8;
	typedef unsigned short     uint16;
	typedef unsigned int       uint32;

#if NV_COMPILER == NV_MSVC
	typedef signed   __int64   sint64;
	typedef unsigned __int64   uint64;
#else
	typedef signed   long long sint64;
	typedef unsigned long long uint64;
#endif
	typedef unsigned char      uchar8;
	typedef unsigned short     uchar16;
	typedef unsigned int       uchar32;

	typedef signed char        schar8;
	typedef signed short       schar16;
	typedef signed long        schar32;

	typedef float              f32;
	typedef double             f64;

	typedef uint64             uid;

	struct empty_type {};

	template < typename T >
	class empty_base_class {};

	template < int a, int b, int c, int d >
	struct four_cc
	{
		static const unsigned int value = (((((d << 8) | c) << 8) | b) << 8) | a;
	};

	class noncopyable
	{
	protected:
		NV_CONSTEXPR noncopyable() = default;
		~noncopyable() = default;
		noncopyable( const noncopyable& ) = delete;
		noncopyable& operator=( const noncopyable& ) = delete;
	};

	template <typename TYPE>
	void construct_object( void* object )
	{
		new (object)TYPE;
	}
	template <typename TYPE>
	void destroy_object( void* object )
	{
		( (TYPE*)object )->TYPE::~TYPE();
	}

} // namespace nv

template <typename OBJ, typename T> 
inline size_t offset_of(T OBJ::*ptr)
{
	return ((size_t)&(((OBJ*)0)->*ptr));
}

template <typename T, typename U>
inline T* down_cast(U* x)
{
#if NV_DEBUG
	T* p = dynamic_cast<T*>(x);
	if (p == 0)
	{
#ifdef NV_LOG
		NV_THROW( std::bad_cast );
#endif
	}

	return p;
#else
	return static_cast<T*>(x);
#endif
}

template <typename T, typename U>
T narrow_cast(const U& a)
{
	return static_cast<T>(a & T(-1) );
}

static_assert( sizeof( nv::sint8 ) == 1, "sint8 size isn't 1 bytes" );
static_assert( sizeof( nv::sint16 ) == 2, "sint16 size isn't 2 bytes" );
static_assert( sizeof( nv::sint32 ) == 4, "sint32 size isn't 4 bytes" );
static_assert( sizeof( nv::sint64 ) == 8, "sint64 size isn't 8 bytes" );

static_assert( sizeof( nv::uint8 ) == 1, "uint8 size isn't 1 byte" );
static_assert( sizeof( nv::uint16 ) == 2, "uint16 size isn't 2 bytes" );
static_assert( sizeof( nv::uint32 ) == 4, "uint32 size isn't 4 bytes" );
static_assert( sizeof( nv::uint64 ) == 8, "uint64 size isn't 8 bytes" );

static_assert( sizeof( nv::f32 ) == 4, "float32 size isn't 4 bytes" );
static_assert( sizeof( nv::f64 ) == 8, "float64 size isn't 8 bytes" );


#endif // NV_CORE_COMMON_HH
