// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file arcball.hh
* @author Kornel Kisielewicz
* @brief arcball utility class
*/

#ifndef NV_CORE_ARCBALL_HH
#define NV_CORE_ARCBALL_HH

#include <nv/core/common.hh>
#include <nv/core/math.hh>

namespace nv
{

	class arcball
	{
	public:
		arcball()
		{
			m_axis  = vec3( 1.0f, 0.0f, 0.0f );
			m_angle = 0.0f;
		}
		arcball( ivec2 window )
			: m_window(window)
		{
			m_axis  = vec3( 1.0f, 0.0f, 0.0f );
			m_angle = 0.0f;
		}
		void set_size( ivec2 window )
		{
			m_window = window;
		}
		void start_drag( ivec2 mposition )
		{
			m_last  = mposition;
			m_axis  = vec3( 1.0f, 0.0f, 0.0f );
			m_angle = 0.0f;
		}
		void drag( ivec2 nposition )
		{
			glm::vec3 va = get_arcball_vector( m_last );
			glm::vec3 vb = get_arcball_vector( nposition );

			m_angle = glm::acos( glm::min( 1.0f, glm::dot( va, vb ) ) );
			m_axis  = glm::cross( va, vb );
			m_last  = nposition;
		}
		const vec3& get_axis() const { return m_axis; }
		f32 get_angle() const { return m_angle; }
	private:
		glm::vec3 get_arcball_vector( const nv::ivec2& s )
		{
			glm::vec3 p( 1.0*s.x / m_window.x * 2 - 1.0, 1.0*s.y / m_window.y * 2 - 1.0, 0 );
			p.y = -p.y;
			float sq = p.x * p.x + p.y * p.y;
			if ( sq <= 1.0f )
				p.z = sqrt( 1.0f - sq );
			else
				p = glm::normalize( p );
			return p;
		}

		ivec2 m_window;
		ivec2 m_last;
		vec3  m_axis;
		f32   m_angle;
	};

}

#endif // NV_CORE_ARCBALL_HH
