// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
* @file memory.hh
* @author Kornel Kisielewicz
* @brief memory utilities
*/

// TODO: implement const and mutable buffer using the union trick

#ifndef NV_CORE_MEMORY_HH
#define NV_CORE_MEMORY_HH

#include <nv/core/common.hh>
#include <nv/stl/type_traits.hh>

namespace nv
{

	namespace detail
	{
		template < typename PARENT, typename T, bool CONST, typename BASE = empty_base_class<PARENT> >
		class pointer_iterators {};

		template < typename PARENT, typename T, typename BASE >
		class pointer_iterators< PARENT, T, true, BASE > : public BASE
		{
		public:
			typedef const T* const_iterator;
			typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

			inline const_iterator begin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator end() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline const_iterator cbegin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator cend() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
			inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
			inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= static_cast<const PARENT&>( *this ).size(), "Index out of range" ); return begin() + i; }
		};

		template < typename PARENT, typename T, typename BASE >
		class pointer_iterators< PARENT, T, false, BASE > : public BASE
		{
		public:
			typedef T*       iterator;
			typedef const T* const_iterator;
			typedef std::reverse_iterator<iterator>       reverse_iterator;
			typedef std::reverse_iterator<const_iterator> const_reverse_iterator;

			inline const_iterator begin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator end() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline iterator begin() { return iterator( static_cast<PARENT&>( *this ).data() ); }
			inline iterator end() { return iterator( static_cast<PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline const_iterator cbegin() const { return const_iterator( static_cast<const PARENT&>( *this ).data() ); }
			inline const_iterator cend() const { return const_iterator( static_cast<const PARENT&>( *this ).data() + static_cast<const PARENT&>( *this ).size() ); }
			inline reverse_iterator rbegin() { return reverse_iterator( end() ); }
			inline const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
			inline const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
			inline reverse_iterator rend() { return reverse_iterator( begin() ); }
			inline const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
			inline const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
			inline const_iterator iat( size_t i ) const { NV_ASSERT( i <= static_cast<const PARENT&>( *this ).size(), "Index out of range" ); return begin() + i; }
			inline iterator       iat( size_t i ) { NV_ASSERT( i <= static_cast<const PARENT&>( *this ).size(), "Index out of range" ); return begin() + i; }
		};

		template < typename T, bool CONST, size_t SIZE >
		class data_base
		{
		};

		template < typename T, size_t SIZE >
		class data_base < T, true, SIZE > 
			: public pointer_iterators< data_base < T, true, SIZE >, T, true > 
		{
		public:
			typedef T                    value_type;
			typedef const value_type*    const_pointer;
			typedef size_t	             size_type;
			typedef const_pointer        const_iterator;
			typedef const value_type&    const_reference;

			inline const_pointer data() const { return m_data; }
			inline size_type size() const { return SIZE; }
			inline bool empty() const { return false; }
			inline size_type   raw_size() const { return SIZE * sizeof( T ); }
			inline const char* raw_data() const { return (const char*)m_data; }

			inline const_reference front() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[0]; }
			inline const_reference back() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[SIZE - 1]; }
		protected:
			value_type m_data[SIZE];
		};

		template < typename T, size_t SIZE >
		class data_base < T, false, SIZE >
			: public pointer_iterators < data_base < T, false, SIZE >, T, false >
		{
		public:
			typedef T                    value_type;
			typedef value_type*          pointer;
			typedef const value_type*    const_pointer;
			typedef size_t	             size_type;
			typedef pointer              iterator;
			typedef const_pointer        const_iterator;
			typedef value_type&		     reference;
			typedef const value_type&    const_reference;

			inline const_pointer data() const { return m_data; }
			inline pointer data() { return m_data; }
			inline size_type size() const { return SIZE; }
			inline bool empty() const { return false; }
			inline size_type   raw_size() const { return SIZE * sizeof( T ); }
			inline const char* raw_data() const { return (const char*)m_data; }
			inline char*       raw_data() { return (char*)m_data; }

			inline reference       front() { NV_ASSERT( !empty(), "front() called on empty data!" );  return m_data[0]; }
			inline const_reference front() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[0]; }
			inline reference       back() { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[SIZE - 1]; }
			inline const_reference back() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[SIZE - 1]; }
		protected:
			value_type m_data[SIZE];
		};


		template < typename T >
		class data_base < T, true, 0 >
			: public pointer_iterators < data_base < T, true, 0 >, T, true >
		{
		public:
			typedef T                    value_type;
			typedef const value_type*    const_pointer;
			typedef size_t	             size_type;
			typedef const_pointer        const_iterator;
			typedef const value_type&    const_reference;

			inline data_base() : m_data( nullptr ), m_size( 0 ) {}
			inline data_base( const_pointer a_data, size_t a_size ) : m_data( a_data ), m_size( a_size ) {}
			inline const T* data() const { return m_data; }
			inline size_t size() const { return m_size; }
			inline bool empty() const { return m_size != 0; }
			inline size_type   raw_size() const { return sizeof( T ) * m_size; }
			inline const char* raw_data() const { return (const char*)m_data; }

			inline const_reference front() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[0]; }
			inline const_reference back() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[m_size - 1]; }
		protected:
			const_pointer m_data;
			size_type     m_size;
		};

		template < typename T >
		class data_base < T, false, 0 >
			: public pointer_iterators < data_base < T, false, 0 >, T, false >
		{
		public:
			typedef T                    value_type;
			typedef value_type*          pointer;
			typedef const value_type*    const_pointer;
			typedef size_t	             size_type;
			typedef pointer              iterator;
			typedef const_pointer        const_iterator;
			typedef value_type&		     reference;
			typedef const value_type&    const_reference;

			inline data_base() : m_data( nullptr ), m_size( 0 ) {}
			inline data_base( pointer a_data, size_t a_size ) : m_data( a_data ), m_size( a_size ) {}
			inline const_pointer data() const { return m_data; }
			inline pointer data() { return m_data; }
			inline size_t size() const { return m_size; }
			inline bool empty() const { return m_size != 0; }
			inline size_type   raw_size() const { return sizeof( T ) * m_size; }
			inline const char* raw_data() const { return (const char*)m_data; }
			inline char*       raw_data() { return (char*)m_data; }

			inline reference       front() { NV_ASSERT( !empty(), "front() called on empty data!" );  return m_data[0]; }
			inline const_reference front() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[0]; }
			inline reference       back() { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[m_size - 1]; }
			inline const_reference back() const { NV_ASSERT( !empty(), "front() called on empty data!" ); return m_data[m_size - 1]; }
		protected:
			pointer   m_data;
			size_type m_size;
		};

	}

	class const_mem_ref : public detail::data_base< char, true, 0 >
	{
	public:
		typedef char                 value_type;
		typedef value_type*          pointer;
		typedef const value_type*    const_pointer;
		typedef value_type&		     reference;
		typedef const value_type&    const_reference;
		typedef size_t	             size_type;
		typedef ptrdiff_t            difference_type;
		typedef const_pointer        const_iterator;
	public:
		inline const_mem_ref() : detail::data_base< char, true, 0 >() {}
		inline const_mem_ref( const void* p, size_type n ) : detail::data_base< char, true, 0 >( (const char*)p, n ) {}
		inline const_mem_ref( const const_mem_ref& l ) : detail::data_base< char, true, 0 >( l.data(), l.size() ) {}
	};

	class mem_ref : public detail::data_base< char, false, 0 >
	{
	public:
		typedef char                 value_type;
		typedef value_type*          pointer;
		typedef const value_type*    const_pointer;
		typedef value_type&		     reference;
		typedef const value_type&    const_reference;
		typedef size_t	             size_type;
		typedef ptrdiff_t            difference_type;
		typedef pointer              iterator;
		typedef const_pointer        const_iterator;
	public:
		inline mem_ref() : detail::data_base< char, false, 0 >() {}
		inline mem_ref( void* p, size_type n ) : detail::data_base< char, false, 0 >( (char*)p, n ) {}
		inline mem_ref( const mem_ref& l ) : detail::data_base< char, false, 0 >( const_cast< char* >( l.data() ), l.size() ) {}
	};


}

#endif // NV_CORE_MEMORY_HH
