// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#ifndef NV_CORE_RANDOM_HH
#define NV_CORE_RANDOM_HH

#include <nv/core/common.hh>
#include <nv/stl/math.hh>
#include <random>

namespace nv
{

	class random
	{
	public:
		typedef std::mt19937::result_type result_type;
		typedef std::mt19937::result_type seed_type;

		random( seed_type seed = 0 );
		seed_type randomize();
		void set_seed( seed_type seed = 0 );
		static random& get();
		result_type rand();
		sint32 srand( sint32 val );
		uint32 urand( uint32 val );
		f32 frand( f32 val );
		sint32 srange( sint32 min, sint32 max );
		uint32 urange( uint32 min, uint32 max );
		f32 frange( f32 min, f32 max );
		uint32 dice( uint32 count, uint32 sides );

		template < typename T >
		glm::detail::tvec2<T> range( glm::detail::tvec2<T> min, glm::detail::tvec2<T> max )
		{
			return glm::detail::tvec2<T>( 
				range_impl( min.x, max.x, std::is_floating_point<T>() ), 
				range_impl( min.y, max.y, std::is_floating_point<T>() )
				);
		}

		template < typename T >
		glm::detail::tvec3<T> range( glm::detail::tvec3<T> min, glm::detail::tvec3<T> max )
		{
			return glm::detail::tvec3<T>( 
				range_impl( min.x, max.x, std::is_floating_point<T>() ), 
				range_impl( min.y, max.y, std::is_floating_point<T>() ),
				range_impl( min.z, max.z, std::is_floating_point<T>() )
				);
		}

		template < typename T >
		glm::detail::tvec4<T> range( glm::detail::tvec4<T> min, glm::detail::tvec4<T> max )
		{
			return glm::detail::tvec4<T>( 
				range_impl( min.x, max.x, std::is_floating_point<T>() ), 
				range_impl( min.y, max.y, std::is_floating_point<T>() ),
				range_impl( min.z, max.z, std::is_floating_point<T>() ), 
				range_impl( min.w, max.w, std::is_floating_point<T>() ) 
				);
		}

		vec3 unit_vec3( bool = false )
		{
			return precise_unit_vec3();
//			return precise ? precise_unit_vec3() : fast_unit_vec3();
		}
		vec2 unit_vec2( bool = false )
		{
			return precise_unit_vec2();
//			return precise ? precise_unit_vec2() : fast_unit_vec2();
		}

		vec2 disk_point( bool precise = false )
		{
			return precise ? precise_disk_point() : fast_disk_point();
		}

		vec3 sphere_point( bool precise = false )
		{
			return precise ? precise_sphere_point() : fast_sphere_point();
		}

		vec2 ellipse_point( const vec2& radii, bool precise = false )
		{
			return precise ? precise_ellipse_point( radii ) : fast_ellipse_point( radii );
		}

		vec3 ellipsoid_point( const vec3& radii, bool precise = false )
		{
			return precise ? precise_ellipsoid_point( radii ) : fast_ellipsoid_point( radii );
		}

		vec2 ellipse_edge( const vec2& radii, bool = false )
		{
			return unit_vec2() * radii;
		}

		vec3 ellipsoid_edge( const vec3& radii, bool = false )
		{
			return unit_vec3() * radii;
		}

		vec2 hollow_disk_point( float iradius, float oradius, bool precise = false )
		{
			return precise ? precise_hollow_disk_point( iradius, oradius ) : fast_hollow_disk_point( iradius, oradius );
		}

		vec3 hollow_sphere_point( float iradius, float oradius, bool precise = false )
		{
			return precise ? precise_hollow_sphere_point( iradius, oradius ) : fast_hollow_sphere_point( iradius, oradius );
		}

		vec2 hollow_ellipse_point( const vec2& iradii, const vec2& oradii, bool precise = false )
		{
			return precise ? precise_hollow_ellipse_point( iradii, oradii ) : fast_hollow_ellipse_point( iradii, oradii );
		}

		vec3 hollow_ellipsoid_point( const vec3& iradii, const vec3& oradii, bool precise = false )
		{
			return precise ? precise_hollow_ellipsoid_point( iradii, oradii ) : fast_hollow_ellipsoid_point( iradii, oradii );
		}

		//vec2 fast_unit_vec2();
		vec2 precise_unit_vec2();
		//vec3 fast_unit_vec3();
		vec3 precise_unit_vec3();
	
		vec2 fast_disk_point();
		vec2 precise_disk_point();
		vec3 fast_sphere_point();
		vec3 precise_sphere_point();

		vec2 fast_ellipse_point( const vec2& radii )
		{
			return fast_disk_point() * radii;
		}
		vec2 precise_ellipse_point( const vec2& radii );

		vec3 fast_ellipsoid_point( const vec3& radii )
		{
			return fast_sphere_point() * radii;
		}

		vec3 precise_ellipsoid_point( const vec3& radii );

		vec2 fast_hollow_disk_point( float iradius, float oradius );
		vec2 precise_hollow_disk_point( float iradius, float oradius );
		vec3 fast_hollow_sphere_point( float iradius, float oradius );
		vec3 precise_hollow_sphere_point( float iradius, float oradius );

		vec2 fast_hollow_ellipse_point( const vec2& iradii, const vec2& oradii );
		vec2 precise_hollow_ellipse_point( const vec2& iradii, const vec2& oradii );
		vec3 fast_hollow_ellipsoid_point( const vec3& iradii, const vec3& oradii );
		vec3 precise_hollow_ellipsoid_point( const vec3& iradii, const vec3& oradii );


	private:
		static seed_type randomized_seed();

		template <typename T>
		T range_impl( T min, T max, const std::true_type& )
		{
			std::uniform_real_distribution<T> dist( min, max );
			return dist( rng );
		}

		template <typename T>
		T range_impl( T min, T max, const std::false_type& )
		{
			std::uniform_int_distribution<T> dist( min, max );
			return dist( rng );
		}
	private:
		std::mt19937 rng;
	};

}

#endif // NV_CORE_RANDOM_HH
