// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file logging.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief logging interfaces
 */

#ifndef NV_CORE_LOGGING_HH
#define NV_CORE_LOGGING_HH

#include <nv/core/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/singleton.hh>
#include <sstream>

#include <nv/core/profiler.hh>



namespace nv
{

	enum log_level
	{
		LOG_NONE     = 0,
		LOG_FATAL    = 10,
		LOG_CRITICAL = 20,
		LOG_ERROR    = 30,
		LOG_WARNING  = 40,
		LOG_NOTICE   = 50,
		LOG_INFO     = 60,
		LOG_DEBUG    = 80,
		LOG_TRACE    = 100
	};

	class logger_base : public singleton< logger_base >
	{
	public:
		logger_base()
		{
			m_pos = m_message;
		}
		unsigned int get_level()
		{
			return m_level;
		}
		void set_level( unsigned int level )
		{
			m_level = level;
		}
		virtual void log( log_level level, const string_ref& message ) = 0;
		void log_append( string_ref ref )
		{
			memcpy( m_pos, ref.data(), ref.size() );
			m_pos += ref.size();
		}
		void log_append( uint32 u )
		{
			m_pos += uint32_to_buffer( u, m_pos );
		}
		void log_append( sint32 s )
		{
			m_pos += sint32_to_buffer( s, m_pos );
		}
		void log_append( f32 f )
		{
			m_pos += f32_to_buffer( f, m_pos );
		}
		void log( log_level level )
		{
			*m_pos = '\0';
			log( level, string_ref( m_message, (size_t) ( m_pos - m_message ) ) );
			m_pos = m_message;
		}
		static bool can_log( log_level level )
		{
			return logger_base::is_valid() && (unsigned int)reference().get_level() >= (unsigned int)level;
		}
		virtual ~logger_base() {}
	protected:
		char m_message[256];
		char* m_pos;
		unsigned int m_level;
	};

	namespace detail
	{
		inline void streamer( std::stringstream& )
		{
			// noop;
		}

		template < typename T, typename... Args >
		inline void streamer( std::stringstream& ss, T&& t, Args&&... args )
		{
			ss << t;
			streamer( ss, std::forward<Args>(args)... );
		}
		inline void logger( log_level level, logger_base& base )
		{
			base.log( level );
		}

		template < typename T, typename... Args >
		inline void logger( log_level level, logger_base& base, T&& t, Args&&... args )
		{
			base.log_append( t );
			logger( level, base, std::forward<Args>( args )... );
		}
	}

} // namespace nv

#define NV_LOG_STREAM(level, message_stream) \
	if ( nv::logger_base::can_log(level) ) \
	{       \
		std::stringstream ss; \
		ss << message_stream;\
		nv::logger_base::reference().log( level, string_ref( ss.str() ) ); \
	}

#define NV_LOG(level, ...) \
	if ( nv::logger_base::can_log(level) ) \
										{       \
		NV_PROFILE( "logging" );\
		nv::detail::logger( level, nv::logger_base::reference(), __VA_ARGS__ ); \
				}

#if NV_DEBUG == 1
#define NV_DEBUG_LOG(level, ...) \
	if ( nv::logger_base::can_log(level) ) \
	{       \
		nv::detail::logger( level, nv::logger_base::reference(), __VA_ARGS__ ); \
	}
#else
#define NV_DEBUG_LOG(level, ...)
#endif

#define NV_LOG_FATAL(...)    NV_LOG( nv::LOG_FATAL, __VA_ARGS__ )
#define NV_LOG_CRITICAL(...) NV_LOG( nv::LOG_CRITICAL, __VA_ARGS__ )
#define NV_LOG_ERROR(...)    NV_LOG( nv::LOG_ERROR, __VA_ARGS__ )
#define NV_LOG_WARNING(...)  NV_LOG( nv::LOG_WARNING, __VA_ARGS__ )
#define NV_LOG_NOTICE(...)   NV_LOG( nv::LOG_NOTICE, __VA_ARGS__ )
#define NV_LOG_INFO(...)     NV_LOG( nv::LOG_INFO, __VA_ARGS__ )
#define NV_LOG_DEBUG(...)    NV_LOG( nv::LOG_DEBUG, __VA_ARGS__ )
#define NV_LOG_TRACE(...)    NV_LOG( nv::LOG_TRACE, __VA_ARGS__ )

#endif // NV_CORE_LOGGING_HH
