// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file logger.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief logger implementation
 */

#ifndef NV_CORE_LOGGER_HH
#define NV_CORE_LOGGER_HH

#include <nv/core/logging.hh>
#include <nv/stl/string.hh>

namespace nv
{
	/**
	 * Class representing log output.
	 *
	 * Base class for all logging output targets.
	 */
	class log_sink
	{
	public:
		/**
		 * Logging function.
		 *
		 * @param level level on which to log.
		 * @param message message to be logged.
		 */
		virtual void log( log_level level, const string_ref& message ) = 0;
		/**
		 * Enforcement of virtual destructor.
		 */
		virtual ~log_sink() {}
	protected:
		/**
		* Optional timestamp string
		*/
		size_t timestamp( char* buffer ) const;
		/**
		* Log level name (unpadded)
		*/
		string_ref level_name( log_level level ) const;
		/**
		* Log level name (padded)
		*/
		string_ref padded_level_name( log_level level ) const;
	};

	/**
	 * Logger class.
	 *
	 * Allows to catch all logging in ORE and in the application using ORE
	 * logging macros. Initially does nothing, needs to have some log sinks
	 * added.
	 */
	class logger : public logger_base
	{
	public:
		/**
		 * Constructor, taking the minimum level for logging.
		 *
		 * No matter what level sinks you add, the passed level is the
		 * overriding cutoff.
		 */
		explicit logger( unsigned int llevel )
		{
			m_level = llevel;
		}

		/**
		 * Logging function.
		 *
		 * Goes over all registered log sinks and prints to them depending
		 * on the log level of the sink. Messages that are lower level than
		 * the initial level passed to the constructor never even reach this
		 * function.
		 *
		 * @param level level on which to log.
		 * @param message message to be logged.
		 */
		virtual void log( log_level level, const string_ref& message );

		/** 
		 * Log sink registering.
		 * 
		 * Registers a new log sink. Ownership is passed to logger - log sink
		 * will be disposed at program shutdown.
		 *
		 * @param sink log sink to be registered.
		 * @param level level on which the sink should log.
		 */
		virtual void add_sink( log_sink* sink, int level );

		/** 
		 * Log sink removal.
		 * 
		 * Removes a log sink. If log sink doesn't exist, nothing will 
		 * happen.
		 *
		 * @param sink log sink to be removed.
		 * @returns true if sink was found, false otherwise
		 */
		virtual bool remove_sink( log_sink* sink );

		/**
		 * Destructor.
		 * 
		 * Also destroys all file sinks.
		 */
		virtual ~logger();

	protected:
		struct log_sink_info
		{
			log_sink* sink;
			uint32    level;
			log_sink_info() : sink( nullptr ), level(0) {}
		};

		/** Log sink list. */
		log_sink_info m_log_sinks[4];
	};

	/**
	 * Console logger sink.
	 *
	 * Logs to std::out -- be sure a console window is open, or std::out redirected!
	 */
	class log_console_sink : public log_sink
	{
	public:
		/**
		 * Log sink constructor
		 */
		log_console_sink( bool coloring = true );
		/**
		 * Logging function.
		 */
		virtual void log( log_level level, const string_ref& message );

	private:
		void* m_handle;
		bool  m_color;
	};

	/**
	* General handle sink.
	*
	* Logs to passed handle.
	*/
	class log_handle_sink : public log_sink
	{
	public:
		/**
		 * Constructor.
		 *
		 * Logs to the passed stream. The stream is NOT disposed at destruction.
		 * Flushing can be controlled by optional flush parameter.
		 *
		 * @param sys_handle system dependent handle to be logged to.
		 * @param flush_always if set to false, wont flush after each line.
		 */
		log_handle_sink( void* sys_handle, bool flush_always )
			: m_handle( sys_handle ), m_flush( flush_always )
		{
		}

		/**
		 * Logging function.
		 */
		virtual void log( log_level level, const string_ref& message );

	protected:
		/** Stored handle. */
		void* m_handle;
		/** Controls flushing. */
		bool m_flush;
	};

	/**
	 * File logger sink.
	 */
	class log_file_sink : public log_handle_sink
	{
	public:
		/**
		* Constructor.
		*
		* Logs to the file passed. File is closed and disposed of after ending.
		* File is not appended, it's overwritten. If file is not createable,
		* the constructor will throw!
		*
		* @param file_name file to be logged to.
		* @param flush_always if set to false, wont flush after each line.
		*/
		log_file_sink( const string_ref& file_name, bool flush_always = true );

		/**
		* Destructor.
		*
		* Flushes, closes file and disposes of the stream.
		*/
		virtual ~log_file_sink();
	};

}

#endif // NV_CORE_LOGGER_HH
