// Copyright (C) 2012-2014 ChaosForge Ltd
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
//
// TODO: tests for string_length
//  * performance tests
//  * matching tests
//  * compatibility tests
//  * check if we can get rid of const templates
//
// String reference implementation based of string_ref proposal and boost::string_ref
//
// http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2012/n3442.html
// http://www.boost.org/doc/libs/1_58_0/libs/utility/doc/html/string_ref.html
//
// TODO: WARNING: this code is centered around 8bit char. In particular, 
//   const_string wont work with non-8bit chars.
// TODO: remove templating on char type, make a string_utf8 instead.

#ifndef NV_CORE_STRING_HH
#define NV_CORE_STRING_HH

#include <string>
#include <algorithm>
#include <cstring>
#include <sstream>
#include <fstream>
#include <nv/core/common.hh>
#include <nv/core/exception.hh>

namespace nv
{
	using std::string;

	/**
	* Utility function for converting any value to string.
	*
	* @param value value to be converted, needs to have << operator
	*        to stream
	* @throw runtime_error Throws runtime_error on conversion fail
	* @return value converted to string
	*/
	template < class T >
	string to_string( const T& value )
	{
		std::stringstream stream;
		stream << value;

		if ( stream.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}

		return stream.str();
	}

	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	inline string to_string( const string& value)
	{
		return value;
	}

	/**
	* Utility function for converting a string to the given type
	*
	* @param vin the string representing the value
	* @param vout the value to be read. Must be streamable with >>
	* @throw runtime_error Throws runtime_error on conversion fail
	*/
	template < class T >
	void from_string( const string& vin, T& vout )
	{
		std::istringstream value( vin );
		value >> vout;

		if ( value.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}
	}

	/**
	* Utility function for converting a string to the given type
	*
	* @param vin the string representing the value
	* @throw runtime_error Throws runtime_error on conversion fail
	*/
	template < class T >
	T string_cast( const string& vin )
	{
		T vout;
		std::istringstream value( vin );
		value >> vout;

		if ( value.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}
		return vout;
	}


	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	inline void from_string( const string& vin, string& vout )
	{
		vout = vin;
	}

	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	template <>
	inline std::string string_cast( const string& vin )
	{
		return vin;
	}


	/**
	* Simple function for slurping a file into a string.
	*/
	inline std::string slurp( const std::string& filename )
	{
		std::ifstream input(filename);
		if ( !input.is_open() ) NV_THROW( std::runtime_error, "File "+filename+" not found!");
		std::stringstream sstr;
		while(input >> sstr.rdbuf());
		return sstr.str();
	}

	inline bool trim( std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		size_t startpos = str.find_first_not_of(" \r\n\t");

		if ( string::npos != endpos || string::npos != startpos )
		{
			if ( string::npos == startpos ) startpos = 0;
			if ( string::npos != endpos )   endpos = endpos+1-startpos;
			str = str.substr( startpos, endpos );
			return true;
		}
		return false;
	}

	inline bool rtrim( std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		if ( string::npos != endpos )
		{
			str = str.substr( 0, endpos+1 );
			return true;
		}
		return false;
	}

	inline bool ltrim( std::string& str )
	{
		size_t startpos = str.find_first_not_of(" \r\n\t");
		if( string::npos != startpos )
		{
			str = str.substr( startpos );
			return true;
		}
		return false;
	}

	inline std::string trimmed( const std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		size_t startpos = str.find_first_not_of(" \r\n\t");

		if ( string::npos != endpos || string::npos != startpos )
		{
			if ( string::npos == startpos ) startpos = 0;
			if ( string::npos != endpos )   endpos = endpos+1-startpos;
			return str.substr( startpos, endpos );
		}
		return str;
	}

	inline std::string rtrimmed( const std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		if ( string::npos != endpos )
		{
			return str.substr( 0, endpos+1 );
		}
		return str;
	}

	inline std::string ltrimmed( const std::string& str )
	{
		size_t startpos = str.find_first_not_of(" \r\n\t");
		if( string::npos != startpos )
		{
			return str.substr( startpos );
		}
		return str;
	}

	inline bool ends_with( const std::string& s, const std::string & ending )
	{
		if ( s.length() >= ending.length() ) {
			return ( 0 == s.compare (s.length() - ending.length(), ending.length(), ending) );
		} else {
			return false;
		}
	}

	inline std::string& remove_chars( std::string& s, const std::string& chars ) 
	{
		s.erase(remove_if(s.begin(), s.end(), 
			[&chars](const char& c) {
				return chars.find(c) != std::string::npos;
			}), s.end());
		return s;
	}

	inline std::string extract_extension( const std::string& filename )
	{
		size_t lastdot = filename.find_last_of( "." );
		std::string ext;
		if ( std::string::npos != lastdot )
			ext = filename.substr( lastdot + 1 );
		std::transform( ext.begin(), ext.end(), ext.begin(), ::tolower );
		return ext;
	}

	inline std::string remove_chars_copy( std::string s, const std::string& chars ) 
	{
		return remove_chars(s, chars);
	}

	namespace detail
	{
		template< typename T >
		struct string_length_impl
		{
			static size_t get( T ) { return 0; }
		};
		template< size_t S >
		struct string_length_impl < char[S] >
		{
			static size_t get( const char[S] ) { return S - 1; }
		};
		template< size_t S >
		struct string_length_impl < const char[S] >
		{
			static size_t get( const char[S] ) { return S - 1; }
		};
		template<>
		struct string_length_impl < char* >
		{
			static size_t get( const char* s ) { return std::strlen( s ); }
		};
		template<>
		struct string_length_impl < const char* >
		{
			static size_t get( const char* s ) { return std::strlen( s ); }
		};
		template<>
		struct string_length_impl < std::string >
		{
			static size_t get( const std::string& s ) { return s.length(); }
		};
		template<>
		struct string_length_impl < const std::string >
		{
			static size_t get( const std::string& s ) { return s.length(); }
		};
	}

	template< typename T >
	using string_length = detail::string_length_impl <
		typename std::remove_cv < typename std::remove_reference< T >::type >::type >;

	class string_ref;

	// string base class - will become a base for a string class later
	class string_base
	{
	public:
		typedef char           value_type;
		typedef const char*    pointer;
		typedef const char&    reference;
		typedef const char&    const_reference;
		typedef pointer        const_iterator;
		typedef const_iterator iterator;
		typedef std::size_t    size_type;
		typedef std::ptrdiff_t difference_type;

		typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
		typedef const_reverse_iterator                reverse_iterator;

		static NV_CONSTEXPR_CONST size_type npos = size_type( -1 );

		// conversion to std::string
 		inline std::string to_string() const
 		{
 			return std::string( m_data, m_size );
 		}

		// iterators
		inline NV_CONSTEXPR const_iterator  cbegin() const { return m_data; }
		inline NV_CONSTEXPR const_iterator    cend() const { return m_data + m_size; }
		inline const_reverse_iterator crbegin() const { return const_reverse_iterator( cend() ); }
		inline const_reverse_iterator   crend() const { return const_reverse_iterator( cbegin() ); }

		inline NV_CONSTEXPR size_type size()     const { return m_size; }
		inline NV_CONSTEXPR size_type length()   const { return m_size; }
		inline NV_CONSTEXPR size_type max_size() const { return m_size; }
		inline NV_CONSTEXPR bool empty()         const { return m_size == 0; }

		// access
		inline NV_CONSTEXPR char operator[]( size_type i ) const { return m_data[i]; }
		inline char at( size_t i ) const
		{
			//	if ( i >= m_data ) NV_THROW( std::out_of_range( "string_ref::at" ) );
			return m_data[i];
		}

		inline NV_CONSTEXPR const char front()  const { return m_data[0]; }
		inline NV_CONSTEXPR const char back()   const { return m_data[m_size - 1]; }
		inline NV_CONSTEXPR const char* data()  const { return m_data; }

		// string operations
		int compare( const string_base& rhs ) const
		{
			int cmp = std::memcmp( m_data, rhs.m_data, ( std::min )( m_size, rhs.m_size ) );
			return cmp != 0 ? cmp : ( m_size == rhs.m_size ? 0 : m_size < rhs.m_size ? -1 : 1 );
		}
		bool starts_with( char c ) const { return !empty() && c == front(); }
		bool starts_with( const string_base& s ) const
		{
			return m_size >= s.m_size && std::memcmp( m_data, s.m_data, s.m_size ) == 0;
		}
		bool ends_with( char c ) const { return !empty() && c == back(); }
		bool ends_with( const string_base& s ) const
		{
			return m_size >= s.m_size && std::memcmp( m_data + m_size - s.m_size, s.m_data, s.m_size ) == 0;
		}
		size_type find( const string_base& s, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_iterator it = std::search( this->cbegin() + pos, this->cend(), s.cbegin(), s.cend() );
			return it == this->cend() ? npos : std::distance( this->cbegin(), it );
		}
		size_type find( char c, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_iterator it = std::find_if( this->cbegin() + pos, this->cend(), [=] ( char val ) { return val == c; } );
			return it == this->cend() ? npos : std::distance( this->cbegin(), it );
		}
		size_type rfind( const string_base& s, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_reverse_iterator it = std::search( this->crbegin() + pos, this->crend(), s.crbegin(), s.crend() );
			return it == this->crend() ? npos : m_size - 1 - std::distance( this->crbegin(), it );
		}
		size_type rfind( char c, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_reverse_iterator it = std::find_if( this->crbegin() + pos, this->crend(), [=] ( char val ) { return val == c; } );
			return it == this->crend() ? npos : m_size - 1 - std::distance( this->crbegin(), it );
		}
		size_type find_first_of( char c ) const { return find( c ); }
		size_type find_first_of( const string_base& s ) const
		{
			const_iterator it = std::find_first_of( this->cbegin(), this->cend(), s.cbegin(), s.cend() );
			return it == this->cend() ? npos : std::distance( this->cbegin(), it );
		}
		size_type find_last_of( char c )  const { return rfind( c ); }
		size_type find_last_of( const string_base& s ) const
		{
			const_reverse_iterator it = std::find_first_of( this->crbegin(), this->crend(), s.cbegin(), s.cend() );
			return it == this->crend() ? npos : m_size - 1 - std::distance( this->crbegin(), it );
		}
		size_type find_first_not_of( const string_base& s ) const
		{
			for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
				if ( 0 == std::memchr( s.m_data, *it, s.m_size ) )
					return std::distance( this->cbegin(), it );
			return npos;
		}
		size_type find_first_not_of( char c ) const
		{
			for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
				if ( c != *it )
					return std::distance( this->cbegin(), it );
			return npos;
		}
		size_type find_last_not_of( const string_base& s ) const
		{
			for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
				if ( 0 == std::memchr( s.m_data, *it, s.m_size ) )
					return m_size - 1 - std::distance( this->crbegin(), it );;
			return npos;
		}
		size_type find_last_not_of( char c ) const
		{
			for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
				if ( c != *it )
					return m_size - 1 - std::distance( this->crbegin(), it );
			return npos;
		}

		// string operations
		string_ref substr( size_type p, size_type n = npos ) const;

		inline std::size_t hash() const
		{
			const char* str = m_data;
			size_type   sz  = m_size;
			int seed = 131;
			int result = 0;
			while ( sz )
			{
				result = ( result * seed ) + ( *str );
				str++;
				sz--;
			}
			return result & ( 0x7FFFFFFF );
		}

	protected:
		inline NV_CONSTEXPR string_base() : m_data( nullptr ), m_size( 0 ) {}
		inline NV_CONSTEXPR string_base( pointer a_data, size_type a_lenght )
			: m_data( a_data ), m_size( a_lenght ) {}

	protected:
		pointer   m_data;
		size_type m_size;
	};

	class string_ref : public string_base
	{
	public:
		inline NV_CONSTEXPR string_ref() {}
		inline NV_CONSTEXPR string_ref( const string_ref& rhs )
			: string_base( rhs.m_data, rhs.m_size )
		{
		}
		inline NV_CONSTEXPR string_ref( const string_base& rhs )
			: string_base( rhs.data(), rhs.size() )
		{
		}

		inline string_ref( const std::string& str )
			: string_base( str.data(), str.size() )
		{
		}

		inline NV_CONSTEXPR string_ref( const char* str, size_type len )
			: string_base( str, len )
		{
		}

		// Literal constructors
		template< size_t N >
		inline string_ref( char( &s )[N] ) : string_base( s, N - 1 ) {}
		template< size_t N >
		inline string_ref( const char( &s )[N] ) : string_base( s, N - 1 ) {}

		// Non-literal constructors
		template< typename U, typename std::enable_if<std::is_same<U, const char*>::value>::type* = nullptr >
		inline string_ref( U str ) : string_base( str, std::strlen( str ) ) {}

		inline string_ref& operator=( const string_ref &rhs )
		{
			m_data = rhs.m_data;
			m_size = rhs.m_size;
			return *this;
		}

		// iterators
		inline NV_CONSTEXPR const_iterator   begin() const { return m_data; }
		inline NV_CONSTEXPR const_iterator     end() const { return m_data + m_size; }
		inline const_reverse_iterator  rbegin() const { return const_reverse_iterator( end() ); }
		inline const_reverse_iterator    rend() const { return const_reverse_iterator( begin() ); }

		// modifiers
		inline void clear()
		{
			m_size = 0;
			m_data = nullptr;
		}
		inline void remove_prefix( size_type n )
		{
			if ( n > m_size )	n = m_size;
			m_data += n;
			m_size -= n;
		}
		inline void remove_suffix( size_type n )
		{
			if ( n > m_size ) n = m_size;
			m_size -= n;
		}

	};

	// const string is movable but not copyable
	class const_string : public string_base
	{
	public:
		NV_CONSTEXPR const_string() {}
		const_string( const char* str, size_type len )
		{
			initialize( str, len );
		}
		explicit const_string( const char* str )
		{
			initialize( str, std::strlen( str ) );
		}

		~const_string()
		{
			if ( m_data ) delete m_data;
		}

		inline const_string( const_string&& other )
		{
			m_data = other.m_data;
			other.m_data = nullptr;
		}

		inline const_string& operator=( const_string&& other )
		{
			pointer temp = m_data;
			m_data = other.m_data;
			other.m_data = temp;
			return *this;
		}

		// blocked copy constructor
		const_string( const const_string & ) = delete;
		// blocked copy operator
		const_string& operator=( const const_string& ) = delete;

	private:

		void initialize( const char* p, size_type s )
		{
			char* data = new char[s + 1];
			std::memcpy( data, p, s );
			data[s] = 0;
			m_data = data;
		}
	};

	inline string_ref string_base::substr( size_type p, size_type n ) const
	{
		if ( p > size() ) return string_ref(); // NV_THROW( std::out_of_range( "substr" ) );
		if ( n == p || p + n > size() ) n = size() - p;
		return string_ref( data() + p, n );
	}

#define NV_STRING_BASE_CAST_OPERATORS( OPERATOR )\
inline bool operator OPERATOR ( const string_base& lhs, const std::string& rhs )\
{\
	return lhs OPERATOR string_ref( rhs );\
}\
inline bool operator OPERATOR ( const std::string& lhs, const string_base& rhs )\
{\
	return string_ref( lhs ) OPERATOR rhs;\
}\
inline bool operator OPERATOR ( const string_base& lhs, const char* rhs )\
{\
	return lhs OPERATOR string_ref( rhs );\
}\
inline bool operator OPERATOR ( const char* lhs, const string_base& rhs )\
{\
	return string_ref( lhs ) OPERATOR rhs;\
}\

	// Base operators
	inline bool operator==( const string_base& lhs, const string_base& rhs )
	{
		return lhs.size() != rhs.size() ? false : ( lhs.compare( rhs ) == 0 );
	}
	inline bool operator!=( const string_base& lhs, const string_base& rhs )
	{
		return lhs.size() != rhs.size() ? true : ( lhs.compare( rhs ) != 0 );
	}
	inline bool operator<( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) < 0;
	}
	inline bool operator>( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) > 0;
	}
	inline bool operator<=( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) <= 0;
	}
	inline bool operator>=( const string_base& lhs, const string_base& rhs )
	{
		return lhs.compare( rhs ) >= 0;
	}

NV_STRING_BASE_CAST_OPERATORS( == )
NV_STRING_BASE_CAST_OPERATORS( != )
NV_STRING_BASE_CAST_OPERATORS( < )
NV_STRING_BASE_CAST_OPERATORS( > )
NV_STRING_BASE_CAST_OPERATORS( <= )
NV_STRING_BASE_CAST_OPERATORS( >= )

	inline std::ostream& operator<<( std::ostream& os, const string_base& str )
	{
		if ( os.good() )
		{
			os.write( str.data(), str.size() );
		}
		return os;
	}

#undef NV_STRING_REF_CAST_OPERATORS


}

namespace std
{

	template<>
	struct hash< nv::string_base >
	{
		std::size_t operator()( const nv::string_base& s ) const
		{
			return s.hash();
		}
	};

}

#endif // NV_CORE_STRING_HH
