// Copyright (C) 2012-2014 ChaosForge Ltd
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh
//
// TODO: tests for string_length
//  * performance tests
//  * matching tests
//  * compatibility tests
//  * check if we can get rid of const templates
//
// String reference implementation based of string_ref proposal and boost::string_ref
//
// http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2012/n3442.html
// http://www.boost.org/doc/libs/1_58_0/libs/utility/doc/html/string_ref.html
//
// TODO: WARNING: this code is centered around 8bit char. In particular, 
//   const_string wont work with non-8bit chars.
// TODO: remove templating on char type and TRAITS, make a string_utf8 instead.

#ifndef NV_CORE_STRING_HH
#define NV_CORE_STRING_HH

#include <string>
#include <algorithm>
#include <cstring>
#include <sstream>
#include <fstream>
#include <nv/core/common.hh>
#include <nv/core/exception.hh>

namespace nv
{
	using std::string;

	/**
	* Utility function for converting any value to string.
	*
	* @param value value to be converted, needs to have << operator
	*        to stream
	* @throw runtime_error Throws runtime_error on conversion fail
	* @return value converted to string
	*/
	template < class T >
	string to_string( const T& value )
	{
		std::stringstream stream;
		stream << value;

		if ( stream.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}

		return stream.str();
	}

	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	inline string to_string( const string& value)
	{
		return value;
	}

	/**
	* Utility function for converting a string to the given type
	*
	* @param vin the string representing the value
	* @param vout the value to be read. Must be streamable with >>
	* @throw runtime_error Throws runtime_error on conversion fail
	*/
	template < class T >
	void from_string( const string& vin, T& vout )
	{
		std::istringstream value( vin );
		value >> vout;

		if ( value.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}
	}

	/**
	* Utility function for converting a string to the given type
	*
	* @param vin the string representing the value
	* @throw runtime_error Throws runtime_error on conversion fail
	*/
	template < class T >
	T string_cast( const string& vin )
	{
		T vout;
		std::istringstream value( vin );
		value >> vout;

		if ( value.fail() )
		{
			NV_THROW( runtime_error, "bad cast" );
		}
		return vout;
	}


	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	inline void from_string( const string& vin, string& vout )
	{
		vout = vin;
	}

	/**
	* Override function for special treatment of strings. Returns the
	* value passed.
	*/
	template <>
	inline std::string string_cast( const string& vin )
	{
		return vin;
	}


	/**
	* Simple function for slurping a file into a string.
	*/
	inline std::string slurp( const std::string& filename )
	{
		std::ifstream input(filename);
		if ( !input.is_open() ) NV_THROW( std::runtime_error, "File "+filename+" not found!");
		std::stringstream sstr;
		while(input >> sstr.rdbuf());
		return sstr.str();
	}

	inline bool trim( std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		size_t startpos = str.find_first_not_of(" \r\n\t");

		if ( string::npos != endpos || string::npos != startpos )
		{
			if ( string::npos == startpos ) startpos = 0;
			if ( string::npos != endpos )   endpos = endpos+1-startpos;
			str = str.substr( startpos, endpos );
			return true;
		}
		return false;
	}

	inline bool rtrim( std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		if ( string::npos != endpos )
		{
			str = str.substr( 0, endpos+1 );
			return true;
		}
		return false;
	}

	inline bool ltrim( std::string& str )
	{
		size_t startpos = str.find_first_not_of(" \r\n\t");
		if( string::npos != startpos )
		{
			str = str.substr( startpos );
			return true;
		}
		return false;
	}

	inline std::string trimmed( const std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		size_t startpos = str.find_first_not_of(" \r\n\t");

		if ( string::npos != endpos || string::npos != startpos )
		{
			if ( string::npos == startpos ) startpos = 0;
			if ( string::npos != endpos )   endpos = endpos+1-startpos;
			return str.substr( startpos, endpos );
		}
		return str;
	}

	inline std::string rtrimmed( const std::string& str )
	{
		size_t endpos = str.find_last_not_of(" \r\n\t");
		if ( string::npos != endpos )
		{
			return str.substr( 0, endpos+1 );
		}
		return str;
	}

	inline std::string ltrimmed( const std::string& str )
	{
		size_t startpos = str.find_first_not_of(" \r\n\t");
		if( string::npos != startpos )
		{
			return str.substr( startpos );
		}
		return str;
	}

	inline bool ends_with( const std::string& s, const std::string & ending )
	{
		if ( s.length() >= ending.length() ) {
			return ( 0 == s.compare (s.length() - ending.length(), ending.length(), ending) );
		} else {
			return false;
		}
	}

	inline std::string& remove_chars( std::string& s, const std::string& chars ) 
	{
		s.erase(remove_if(s.begin(), s.end(), 
			[&chars](const char& c) {
				return chars.find(c) != std::string::npos;
			}), s.end());
		return s;
	}

	inline std::string extract_extension( const std::string& filename )
	{
		size_t lastdot = filename.find_last_of( "." );
		std::string ext;
		if ( std::string::npos != lastdot )
			ext = filename.substr( lastdot + 1 );
		std::transform( ext.begin(), ext.end(), ext.begin(), ::tolower );
		return ext;
	}

	inline std::string remove_chars_copy( std::string s, const std::string& chars ) 
	{
		return remove_chars(s, chars);
	}

	namespace detail
	{
		template< typename T >
		struct string_length_impl
		{
			static size_t get( T ) { return 0; }
		};
		template< size_t S >
		struct string_length_impl < char[S] >
		{
			static size_t get( const char[S] ) { return S - 1; }
		};
		template< size_t S >
		struct string_length_impl < const char[S] >
		{
			static size_t get( const char[S] ) { return S - 1; }
		};
		template<>
		struct string_length_impl < char* >
		{
			static size_t get( const char* s ) { return std::strlen( s ); }
		};
		template<>
		struct string_length_impl < const char* >
		{
			static size_t get( const char* s ) { return std::strlen( s ); }
		};
		template<>
		struct string_length_impl < std::string >
		{
			static size_t get( const std::string& s ) { return s.length(); }
		};
		template<>
		struct string_length_impl < const std::string >
		{
			static size_t get( const std::string& s ) { return s.length(); }
		};
	}

	template< typename T >
	using string_length = detail::string_length_impl <
		typename std::remove_cv < typename std::remove_reference< T >::type >::type >;

	template<typename T, typename TRAITS = std::char_traits<T> > class basic_string_ref;
	
	// string base class - will become a base for a string class later
	template<typename T, typename TRAITS = std::char_traits<T> >
	class basic_string_base
	{
	public:
		typedef T              value_type;
		typedef const T*       pointer;
		typedef const T&       reference;
		typedef const T&       const_reference;
		typedef pointer        const_iterator;
		typedef const_iterator iterator;
		typedef std::size_t    size_type;
		typedef std::ptrdiff_t difference_type;

		typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
		typedef const_reverse_iterator                reverse_iterator;

		static NV_CONSTEXPR_CONST size_type npos = size_type( -1 );

		// conversion to std::string
 		std::basic_string<T, TRAITS> to_string() const
 		{
 			return std::basic_string<T, TRAITS>( m_data, m_size );
 		}

		// iterators
		NV_CONSTEXPR const_iterator  cbegin() const { return m_data; }
		NV_CONSTEXPR const_iterator    cend() const { return m_data + m_size; }
		const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
		const_reverse_iterator   crend() const { return const_reverse_iterator( begin() ); }

		NV_CONSTEXPR size_type size()     const { return m_size; }
		NV_CONSTEXPR size_type length()   const { return m_size; }
		NV_CONSTEXPR size_type max_size() const { return m_size; }
		NV_CONSTEXPR bool empty()         const { return m_size == 0; }

		// access
		NV_CONSTEXPR const T& operator[]( size_type i ) const { return m_data[i]; }
		const T& at( size_t i ) const
		{
			//	if ( i >= m_data ) NV_THROW( std::out_of_range( "string_ref::at" ) );
			return m_data[i];
		}

		NV_CONSTEXPR const T& front() const { return m_data[0]; }
		NV_CONSTEXPR const T& back()  const { return m_data[m_size - 1]; }
		NV_CONSTEXPR const T* data()  const { return m_data; }

		// string operations
		int compare( basic_string_base rhs ) const
		{
			int cmp = TRAITS::compare( m_data, rhs.m_data, ( std::min )( m_size, rhs.m_size ) );
			return cmp != 0 ? cmp : ( m_size == rhs.m_size ? 0 : m_size < rhs.m_size ? -1 : 1 );
		}
		bool starts_with( T c ) const { return !empty() && TRAITS::eq( c, front() ); }
		bool starts_with( basic_string_base s ) const
		{
			return m_size >= s.m_size && TRAITS::compare( m_data, s.m_data, s.m_size ) == 0;
		}
		bool ends_with( T c ) const { return !empty() && TRAITS::eq( c, back() ); }
		bool ends_with( basic_string_base s ) const
		{
			return m_size >= s.m_size && TRAITS::compare( m_data + m_size - s.m_size, s.m_data, s.m_size ) == 0;
		}
		size_type find( basic_string_base s, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_iterator it = std::search( this->cbegin() + pos, this->cend(), s.cbegin(), s.cend(), TRAITS::eq );
			return it == this->cend() ? npos : std::distance( this->cbegin(), it );
		}
		size_type find( T c, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_iterator it = std::find_if( this->cbegin() + pos, this->cend(), [=] ( T val ) { return TRAITS::eq( val, c ); } );
			return it == this->cend() ? npos : std::distance( this->cbegin(), it );
		}
		size_type rfind( basic_string_base s, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_reverse_iterator it = std::search( this->crbegin() + pos, this->crend(), s.crbegin(), s.crend(), TRAITS::eq );
			return it == this->crend() ? npos : m_size - 1 - std::distance( this->crbegin(), it );
		}
		size_type rfind( T c, size_type pos = 0 ) const
		{
			if ( pos >= m_size ) return npos;
			const_reverse_iterator it = std::find_if( this->crbegin() + pos, this->crend(), [=] ( T val ) { return TRAITS::eq( val, c ); } );
			return it == this->crend() ? npos : m_size - 1 - std::distance( this->crbegin(), it );
		}
		size_type find_first_of( T c ) const { return find( c ); }
		size_type find_first_of( basic_string_base s ) const
		{
			const_iterator it = std::find_first_of( this->cbegin(), this->cend(), s.cbegin(), s.cend(), TRAITS::eq );
			return it == this->cend() ? npos : std::distance( this->cbegin(), it );
		}
		size_type find_last_of( T c )  const { return rfind( c ); }
		size_type find_last_of( basic_string_base s ) const
		{
			const_reverse_iterator it = std::find_first_of( this->crbegin(), this->crend(), s.cbegin(), s.cend(), TRAITS::eq );
			return it == this->crend() ? npos : m_size - 1 - std::distance( this->crbegin(), it );
		}
		size_type find_first_not_of( basic_string_base s ) const
		{
			for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
				if ( 0 == TRAITS::find( s.m_data, s.m_size, *first ) )
					return std::distance( this->cbegin(), it );
			return npos;
		}
		size_type find_first_not_of( T c ) const
		{
			for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
				if ( !TRAITS::eq( c, *it ) )
					return std::distance( this->cbegin(), it );
			return npos;
		}
		size_type find_last_not_of( basic_string_base s ) const
		{
			for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
				if ( 0 == TRAITS::find( s.m_data, s.m_size, *it ) )
					return m_size - 1 - std::distance( this->crbegin(), it );;
			return npos;
		}
		size_type find_last_not_of( T c ) const
		{
			for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
				if ( !TRAITS::eq( c, *it ) )
					return m_size - 1 - std::distance( this->crbegin(), it );
			return npos;
		}

		// string operations
		basic_string_ref< T, TRAITS > substr( size_type p, size_type n = npos ) const
		{
			if ( p > size() ) return basic_string_ref< T, TRAITS >(); // NV_THROW( std::out_of_range( "substr" ) );
			if ( n == p || p + n > size() ) n = size() - p;
			return basic_string_ref< T, TRAITS >( data() + p, n );
		}

	protected:
		NV_CONSTEXPR basic_string_base() : m_data( nullptr ), m_size( 0 ) {}
		NV_CONSTEXPR basic_string_base( pointer a_data, size_type a_lenght )
			: m_data( a_data ), m_size( a_lenght ) {}

	protected:
		pointer   m_data;
		size_type m_size;
	};

#define NV_STRING_BASE_CAST_OPERATORS( OPERATOR )\
template < typename T, typename TRAITS, typename ALLOCATOR >\
inline bool operator OPERATOR ( basic_string_base< T, TRAITS > lhs, const std::basic_string< T, TRAITS, ALLOCATOR >& rhs )\
		{\
	return lhs OPERATOR basic_string_base< T, TRAITS >( rhs );\
		}\
template < typename T, typename TRAITS, typename ALLOCATOR >\
inline bool operator OPERATOR ( const std::basic_string< T, TRAITS, ALLOCATOR >& lhs, basic_string_base< T, TRAITS > rhs )\
		{\
	return basic_string_base< T, TRAITS >( lhs ) OPERATOR rhs;\
		}\
template < typename T, typename TRAITS >\
inline bool operator OPERATOR ( basic_string_base<T, TRAITS> lhs, const T* rhs )\
		{\
	return lhs OPERATOR basic_string_base< T, TRAITS >( rhs );\
		}\
template < typename T, typename TRAITS >\
inline bool operator OPERATOR ( const T* lhs, basic_string_base< T, TRAITS > rhs )\
		{\
	return basic_string_base< T, TRAITS >( lhs ) OPERATOR rhs;\
		}\

	// Base operators
	template < typename T, typename TRAITS >
	inline bool operator==( basic_string_base< T, TRAITS > lhs, basic_string_base< T, TRAITS > rhs )
	{
		return lhs.size() != rhs.size() ? false : ( lhs.compare( rhs ) == 0 );
	}
	template < typename T, typename TRAITS >
	inline bool operator!=( basic_string_base< T, TRAITS > lhs, basic_string_base< T, TRAITS > rhs )
	{
		return lhs.size() != rhs.size() ? true : ( lhs.compare( rhs ) != 0 );
	}
	template < typename T, typename TRAITS >
	inline bool operator<( basic_string_base< T, TRAITS > lhs, basic_string_base< T, TRAITS > rhs )
	{
		return lhs.compare( rhs ) < 0;
	}
	template < typename T, typename TRAITS >
	inline bool operator>( basic_string_base< T, TRAITS > lhs, basic_string_base< T, TRAITS > rhs )
	{
		return lhs.compare( rhs ) > 0;
	}
	template < typename T, typename TRAITS >
	inline bool operator<=( basic_string_base< T, TRAITS > lhs, basic_string_base< T, TRAITS > rhs )
	{
		return lhs.compare( rhs ) <= 0;
	}
	template < typename T, typename TRAITS >
	inline bool operator>=( basic_string_base< T, TRAITS > lhs, basic_string_base< T, TRAITS > rhs )
	{
		return lhs.compare( rhs ) >= 0;
	}

	NV_STRING_BASE_CAST_OPERATORS( == )
	NV_STRING_BASE_CAST_OPERATORS( != )
	NV_STRING_BASE_CAST_OPERATORS( < )
	NV_STRING_BASE_CAST_OPERATORS( > )
	NV_STRING_BASE_CAST_OPERATORS( <= )
	NV_STRING_BASE_CAST_OPERATORS( >= )

	template < class T, class TRAITS >
	inline std::basic_ostream<T, TRAITS>& operator<<( std::basic_ostream<T, TRAITS>& os, const basic_string_base<T, TRAITS>& str )
	{
		if ( os.good() )
		{
			os.write( str.data(), str.size() );
		}
		return os;
	}

#undef NV_STRING_REF_CAST_OPERATORS

	template<typename T, typename TRAITS>
	class basic_string_ref : public basic_string_base < T, TRAITS >
	{
	public:

		NV_CONSTEXPR basic_string_ref() {}
		NV_CONSTEXPR basic_string_ref( const basic_string_ref &rhs )
			: basic_string_base( rhs.m_data, rhs.m_size )
		{
		}
		NV_CONSTEXPR basic_string_ref( const basic_string_base< T, TRAITS > &rhs )
			: basic_string_base( rhs.m_data, rhs.m_size )
		{
		}

		template< typename ALLOCATOR >
		basic_string_ref( const std::basic_string<T, TRAITS, ALLOCATOR>& str )
			: basic_string_base( str.data(), str.length() )
		{
		}

		NV_CONSTEXPR basic_string_ref( const T* str, size_type len )
			: basic_string_base( str, len )
		{
		}

		// Literal constructors
		template< size_t N, typename std::enable_if<std::is_same<T, char>::value>::type* = nullptr >
		basic_string_ref( T( &s )[N] ) : basic_string_base( s, N - 1 ) {}
		template< size_t N, typename std::enable_if<std::is_same<T, char>::value>::type* = nullptr >
		basic_string_ref( const T( &s )[N] ) : basic_string_base( s, N - 1 ) {}

		// Non-literal constructors
		template< typename U, typename std::enable_if<std::is_same<U, const char*>::value>::type* = nullptr >
		basic_string_ref( U str ) : basic_string_base( str, TRAITS::length( str ) ) {}

		basic_string_ref& operator=( const basic_string_ref &rhs )
		{
			m_data = rhs.m_data;
			m_size = rhs.m_size;
			return *this;
		}

		// iterators
		NV_CONSTEXPR const_iterator   begin() const { return m_data; }
		NV_CONSTEXPR const_iterator     end() const { return m_data + m_size; }
		const_reverse_iterator  rbegin() const { return const_reverse_iterator( end() ); }
		const_reverse_iterator    rend() const { return const_reverse_iterator( begin() ); }

		// modifiers
		void clear()
		{
			m_size = 0;
			m_data = nullptr;
		}
		void remove_prefix( size_type n )
		{
			if ( n > m_size )	n = m_size;
			m_data += n;
			m_size -= n;
		}
		void remove_suffix( size_type n )
		{
			if ( n > m_size ) n = m_size;
			m_size -= n;
		}

	};

	typedef basic_string_ref<char, std::char_traits<char> > string_ref;

	// const string is movable but not copyable
	template<typename T, typename TRAITS>
	class basic_const_string : public basic_string_base < T, TRAITS >
	{
	public:
		basic_const_string() {}
		basic_const_string( const T* str, size_type len )
		{
			initialize( str, len );
		}
		explicit basic_const_string( const T* str )
		{
			initialize( str, TRAITS::length( str ) );
		}

		~basic_const_string()
		{
			if ( m_data ) delete m_data;
		}

		basic_const_string( const basic_const_string&& other )
		{
			std::swap( m_data, other.m_data );
			return *this;
		}

		basic_const_string& operator=( const basic_const_string&& other )
		{
			std::swap( m_data, other.m_data );
			return *this;
		}

	private:
		// blocked copy constructor
		basic_const_string( const basic_const_string & );
		// blocked copy operator
		basic_const_string& operator=( const basic_const_string& );

		void assign( const T* p, size_type s )
		{
			m_data = new T[s + 1];
			TRAITS::copy( m_data, p, s );
			m_data[s] = 0;
		}
	};


}

#endif // NV_CORE_STRING_HH
