// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

/**
 * @file uid.hh
 * @author Kornel Kisielewicz
 * @brief Implements a unique identifier class manager
 */

#ifndef NV_CORE_UID_HH
#define NV_CORE_UID_HH

#include <nv/core/common.hh>
#include <unordered_map>

namespace nv
{

	class uid_store_raw
	{
	public:
		
		/**
		 * Creates a new instance of the unique identifier store.
		 */
		uid_store_raw();

		/**
		 * Gets the object with the specified ID.
		 *
		 * @param auid The ID to fetch.
		 * @returns The stored object for that ID.
		 */
		void* get( uid auid ) const;

		/**
		 * Removes the object with the specified ID.
		 *
		 * @param auid The ID to remove.
		 * @returns True if the removal was successful, false if the ID didn't exist.
		 */
		bool remove( uid auid );

		/**
		 * Adds an object to the store assigned to the indicated ID.
		 *
		 * @param o The object to add to the store.
		 * @param auid The ID to assign to the object.
		 */
		void insert( void* o, uid auid );

		/**
		 * Adds an object to the store and assigns it a new ID.
		 *
		 * @param o The object to add to the store.
		 * @returns The ID the object was store under.
		 */
		uid insert( void* o );

		/**
		 * Gets the next available ID.
		 *
		 * @returns The next ID in the store.
		 */
		uid request_uid();

		/**
		 * Destroys the unique identifier store.
		 */
		~uid_store_raw();

	protected:
		typedef std::unordered_map< uid, void* > map;
		map m_map; ///< The hash map everything is stored in.
		uid m_current; ///< The last UID assigned.
	};

	template < typename T >
	class uid_store
	{
	public:
		T* get( uid auid ) const      { return static_cast<T*>( m_store.get( auid ) ); }
		bool remove( uid auid )       { return m_store.remove( auid ); }
		void insert( T* o, uid auid ) { m_store.insert( o, auid ); }
		uid insert( T* o )            { return m_store.insert( o ); }
		uid request_uid()             { return m_store.request_uid(); }

		/**
		 * Retrieves an object and casts it to the specified type.
		 *
		 * @tparam U The type to cast to.
		 * @param auid The ID the object is stored under.
		 * @returns An object of the indicated type that was stored at the indicated ID.
		 */
		template< typename U >
		U* get_as( uid auid ) const 
		{
			return dynamic_cast< U* >( get(auid) );
		}
	protected:
		uid_store_raw m_store;
	};
	
}

#endif // NV_CORE_UID_HH
